# WVC Theme CSS Build System

This theme uses **Tailwind CSS** with a modern build system that automatically compiles your styles from `assets/css/app.css` into the WordPress theme's `style.css` file.

## 🚀 Quick Start

### Option 1: Automatic Setup (Recommended)
**Run the comprehensive setup script** that handles everything:
```bash
bash maintenance/setup.sh
```

This script automatically:
- ✅ Checks system requirements (PHP, Node.js, npm, Git)
- ✅ Installs Composer (if not present)
- ✅ Installs all PHP development dependencies
- ✅ Configures PHP CodeSniffer with WordPress standards
- ✅ Installs npm dependencies for CSS build system
- ✅ Sets up Git hooks for code quality and CSS building
- ✅ Runs initial CSS build and verification tests

### Option 2: Manual Setup
If you prefer to set up components individually:

1. **Install dependencies**: Follow the setup steps in [readme.md](readme.md)
2. **Setup CSS build system**: `cd wvc-theme/build && npm install`
3. **Start developing**:
   ```bash
   cd wvc-theme/build
   npm run dev  # Watch mode - rebuilds on changes
   ```

## 📁 File Structure

```
wvc-theme/
├── build/
│   ├── build-assets/css/     # 📁 Generated CSS files
│   ├── build-components/     # 📁 React components
│   ├── package.json          # 📦 npm dependencies
│   ├── tailwind.config.js    # ⚙️  Tailwind configuration
│   ├── postcss.config.js     # ⚙️  PostCSS configuration
│   └── build-css.sh          # 🔨 Build script
├── maintenance/
│   ├── setup.sh              # 🚀 Comprehensive setup script
│   └── setup_git_hooks.sh    # 🪝 Git hooks setup
└── style.css                 # ✨ Generated WordPress theme stylesheet
```

## 🔧 Build Commands

| Command | Description |
|---------|-------------|
| `npm run build-css` | Build CSS once |
| `npm run watch-css` | Watch and rebuild CSS automatically |
| `npm run dev` | Same as watch-css |
| `./build-css.sh` | Run build script directly |

## 🎨 How It Works

### 1. **Source File**: `assets/css/app.css`
This is your main CSS file that imports:
- Tailwind CSS (`@tailwind base`, `@tailwind components`, `@tailwind utilities`)
- Your CSS variables (`./variables.css`)
- Your component styles (`./components.css`)
- Custom styles organized in Tailwind layers

### 2. **Build Process**
The build process uses PostCSS with Tailwind CSS to:
- Process Tailwind directives
- Import CSS files
- Apply autoprefixer for browser compatibility
- Generate the final `style.css` with WordPress theme headers

### 3. **Pre-commit Hook**
Automatically builds CSS when you commit changes to:
- Any `.css`, `.scss`, or `.sass` files
- `tailwind.config.js`
- `postcss.config.js`
- Files in `assets/css/` directory

## ✨ Tailwind Configuration

The `tailwind.config.js` is pre-configured to:
- ✅ Scan all PHP files for Tailwind classes
- ✅ Map your CSS variables to Tailwind utilities
- ✅ Include the Typography plugin
- ✅ Support all your existing design tokens

### Using CSS Variables with Tailwind

Your existing CSS variables are mapped to Tailwind utilities:

```html
<!-- These are equivalent: -->
<div class="text-primary">Using Tailwind class</div>
<div style="color: var(--color-primary)">Using CSS variable</div>

<!-- More examples: -->
<div class="bg-surface p-md rounded-lg shadow-sm">
  <h2 class="text-h2 font-bold text-heading mb-sm">Title</h2>
  <p class="text-base leading-relaxed text-text-secondary">Content</p>
</div>
```

## 🪝 Pre-commit Hook

The pre-commit hook automatically:
1. 🔍 Detects changes to CSS or config files
2. 🔨 Runs the build process
3. ✅ Adds the generated `style.css` to your commit
4. ❌ Prevents commit if build fails

To manually install the pre-commit hook:
```bash
cp .git-hooks/pre-commit .git/hooks/pre-commit
chmod +x .git/hooks/pre-commit
```

## 🎯 Best Practices

### 1. **Layer Organization**
Use Tailwind layers to organize your styles:

```css
@layer base {
  /* Base styles for HTML elements */
  body { ... }
  h1, h2, h3 { ... }
}

@layer components {
  /* Reusable component styles */
  .btn { ... }
  .card { ... }
}

@layer utilities {
  /* Custom utility classes */
  .text-balance { ... }
}
```

### 2. **CSS Variables Integration**
Continue using your CSS variables for consistency:

```css
@layer components {
  .custom-component {
    background-color: var(--color-surface);
    padding: var(--spacing-md);
    border-radius: var(--border-radius-lg);
  }
}
```

### 3. **Responsive Design**
Use Tailwind's responsive utilities:

```html
<div class="container mx-auto px-md lg:px-lg">
  <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-md">
    <!-- Content -->
  </div>
</div>
```

## 🚨 Troubleshooting

### Build Fails
1. Check Node.js version (requires Node 14+)
2. Delete `node_modules` and run `npm install`
3. Check for syntax errors in CSS files

### Pre-commit Hook Not Working
1. Verify hook is executable: `chmod +x .git/hooks/pre-commit`
2. Check git hooks are enabled: `git config core.hooksPath`
3. Run hook manually: `.git/hooks/pre-commit`

### Styles Not Appearing
1. Check `style.css` was generated and committed
2. Clear browser cache
3. Verify WordPress is loading the correct stylesheet

## 📝 Development Workflow

### First Time Setup
1. **Run setup script**: `bash maintenance/setup.sh` (handles everything automatically)
2. **Or manual setup**: Follow the Quick Start → Option 2 above

### Daily Development
1. **Edit** CSS files in `wvc-theme/build/build-assets/css/`
2. **Watch** changes with `cd wvc-theme/build && npm run dev`
3. **Test** in browser (styles rebuild automatically)
4. **Commit** changes (CSS builds automatically via pre-commit hook)
5. **Deploy** (include the generated `style.css`)

## 🔄 Migration from Old System

If you're migrating from the old manual CSS system:

1. Your existing `variables.css` and `components.css` are preserved
2. The main `style.css` is now generated (don't edit directly)
3. All your existing styles are maintained
4. You can now also use Tailwind utility classes
5. The WordPress theme header is preserved in the generated file

---

Happy coding! 🎉 