<?php

namespace WVC\CartHandler;

/**
 * WooCommerce Setup
 *
 * Handles automatic creation and configuration of WooCommerce pages
 *
 * @package    WVC_Theme
 * @subpackage CartHandler
 * @author     10Web
 * @since      1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * Setup WooCommerce pages if they don't exist
 */
function wvc_setup_woocommerce_pages() {
    // Only run if WooCommerce is active
    if ( ! class_exists( 'WooCommerce' ) ) {
        return;
    }

    error_log( "WVC: Setting up WooCommerce pages..." ); 

    $pages = array(
        
        'cart' => array(
            'name'    => 'cart',
            'title'   => 'Cart',
            'content' => '[woocommerce_cart]',
            'option'  => 'woocommerce_cart_page_id'
        ),
        'checkout' => array(
            'name'    => 'checkout',
            'title'   => 'Checkout',
            'content' => '[woocommerce_checkout]',
            'option'  => 'woocommerce_checkout_page_id'
        ),
    );

    $created_pages = array();

    foreach ( $pages as $key => $page ) {
        $page_id = get_option( $page['option'] );
        
        // Check if page exists and is published
        if ( ! $page_id || get_post_status( $page_id ) !== 'publish' ) {
            // Create the page
            $new_page_id = wp_insert_post( array(
                'post_title'   => $page['title'],
                'post_name'    => $page['name'],
                'post_content' => $page['content'],
                'post_status'  => 'publish',
                'post_type'    => 'page',
                'post_author'  => 1,
            ) );

            if ( $new_page_id && ! is_wp_error( $new_page_id ) ) {
                // Update WooCommerce option with new page ID
                update_option( $page['option'], $new_page_id );
                $created_pages[] = $page['title'];
                error_log( "WVC: Created WooCommerce {$page['title']} page (ID: {$new_page_id})" );
            }
        }
    }
    
    // Flush rewrite rules after creating pages
    if ( ! empty( $created_pages ) ) {
        flush_rewrite_rules();
        
        // Set a transient to show success message
        set_transient( 'wvc_woo_pages_created', $created_pages, 45 );
    }
}

/**
 * Check if WooCommerce pages are missing
 */
function wvc_check_woocommerce_pages_missing() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        return false;
    }

    $required_pages = array(
        'woocommerce_cart_page_id',
        'woocommerce_checkout_page_id'
    );

    foreach ( $required_pages as $option ) {
        $page_id = get_option( $option );
        if ( ! $page_id || get_post_status( $page_id ) !== 'publish' ) {
            return true;
        }
    }

    return false;
}


