<?php
/**
 * WVC Plugin Manager
 *
 * @since      1.0.0
 * @package    WVC_Editor
 * @subpackage WVC_Editor/Managers
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * WVC Plugin Manager Class
 */
class WVC_Plugin_Manager
{
    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        // Constructor logic if needed
    }

    /**
     * Install and activate a plugin
     *
     * @param string $plugin_slug Plugin slug to install
     * @return array Operation result
     */
    public function install_plugin($plugin_slug)
    {
        try {
            // Sanitize plugin slug
            $plugin_slug = sanitize_text_field($plugin_slug);

            // Check if plugin is already installed
            $plugin_file = $this->get_plugin_file($plugin_slug);
            if ($plugin_file && file_exists(WP_PLUGIN_DIR . '/' . $plugin_file)) {
                // Plugin already installed, check if it's active
                if (is_plugin_active($plugin_file)) {
                    return [
                        'success' => true,
                        'message' => "Plugin '{$plugin_slug}' is already installed and active",
                        'plugin_slug' => $plugin_slug,
                        'status' => 'active'
                    ];
                } else {
                    // Activate the existing plugin
                    $result = activate_plugin($plugin_file);
                    if (is_wp_error($result)) {
                        return [
                            'success' => false,
                            'message' => "Failed to activate plugin '{$plugin_slug}': " . $result->get_error_message(),
                            'plugin_slug' => $plugin_slug
                        ];
                    }
                    return [
                        'success' => true,
                        'message' => "Plugin '{$plugin_slug}' was already installed, now activated",
                        'plugin_slug' => $plugin_slug,
                        'status' => 'activated'
                    ];
                }
            }

            // Install the plugin
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/plugin-install.php';

            $api = plugins_api('plugin_information', [
                'slug' => $plugin_slug,
                'fields' => ['sections' => false]
            ]);

            if (is_wp_error($api)) {
                return [
                    'success' => false,
                    'message' => "Plugin '{$plugin_slug}' not found in repository: " . $api->get_error_message(),
                    'plugin_slug' => $plugin_slug
                ];
            }

            // Create a quiet upgrader skin for CLI/REST operations
            $upgrader_skin = new class extends WP_Upgrader_Skin {
                public function header() {}
                public function footer() {}
                public function feedback($string, ...$args) {}
            };

            $upgrader = new Plugin_Upgrader($upgrader_skin);
            $result = $upgrader->install($api->download_link);

            if (is_wp_error($result)) {
                return [
                    'success' => false,
                    'message' => "Failed to install plugin '{$plugin_slug}': " . $result->get_error_message(),
                    'plugin_slug' => $plugin_slug
                ];
            }

            if ($result === false) {
                return [
                    'success' => false,
                    'message' => "Failed to install plugin '{$plugin_slug}'",
                    'plugin_slug' => $plugin_slug
                ];
            }

            // Activate the newly installed plugin
            $plugin_file = $this->get_plugin_file($plugin_slug);
            if ($plugin_file) {
                $activate_result = activate_plugin($plugin_file);
                if (is_wp_error($activate_result)) {
                    return [
                        'success' => true,
                        'message' => "Plugin '{$plugin_slug}' installed but failed to activate: " . $activate_result->get_error_message(),
                        'plugin_slug' => $plugin_slug,
                        'status' => 'installed_not_activated'
                    ];
                }
            }

            return [
                'success' => true,
                'message' => "Plugin '{$plugin_slug}' installed and activated successfully",
                'plugin_slug' => $plugin_slug,
                'status' => 'installed_and_activated'
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "Error installing plugin '{$plugin_slug}': " . $e->getMessage(),
                'plugin_slug' => $plugin_slug
            ];
        }
    }

    /**
     * Uninstall a plugin
     *
     * @param string $plugin_slug Plugin slug to uninstall
     * @return array Operation result
     */
    public function uninstall_plugin($plugin_slug)
    {
        try {
            // Sanitize plugin slug
            $plugin_slug = sanitize_text_field($plugin_slug);

            $plugin_file = $this->get_plugin_file($plugin_slug);

            if (!$plugin_file) {
                return [
                    'success' => false,
                    'message' => "Plugin '{$plugin_slug}' is not installed",
                    'plugin_slug' => $plugin_slug
                ];
            }

            // Deactivate the plugin first
            if (is_plugin_active($plugin_file)) {
                deactivate_plugins($plugin_file);
            }

            // Delete the plugin
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/plugin.php';

            $result = delete_plugins([$plugin_file]);

            if (is_wp_error($result)) {
                return [
                    'success' => false,
                    'message' => "Failed to uninstall plugin '{$plugin_slug}': " . $result->get_error_message(),
                    'plugin_slug' => $plugin_slug
                ];
            }

            if ($result === false) {
                return [
                    'success' => false,
                    'message' => "Failed to uninstall plugin '{$plugin_slug}'",
                    'plugin_slug' => $plugin_slug
                ];
            }

            return [
                'success' => true,
                'message' => "Plugin '{$plugin_slug}' uninstalled successfully",
                'plugin_slug' => $plugin_slug,
                'status' => 'uninstalled'
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "Error uninstalling plugin '{$plugin_slug}': " . $e->getMessage(),
                'plugin_slug' => $plugin_slug
            ];
        }
    }

    /**
     * List all installed plugins
     *
     * @return array List of plugins
     */
    public function list_plugins()
    {
        try {
            if (!function_exists('get_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }

            $plugins = get_plugins();
            $plugin_list = [];

            foreach ($plugins as $plugin_file => $plugin_data) {
                $plugin_slug = dirname($plugin_file);
                if ($plugin_slug === '.') {
                    $plugin_slug = basename($plugin_file, '.php');
                }

                $plugin_list[] = [
                    'slug' => $plugin_slug,
                    'file' => $plugin_file,
                    'name' => $plugin_data['Name'],
                    'version' => $plugin_data['Version'],
                    'description' => $plugin_data['Description'],
                    'author' => $plugin_data['Author'],
                    'active' => is_plugin_active($plugin_file),
                    'network_active' => is_plugin_active_for_network($plugin_file)
                ];
            }

            return [
                'success' => true,
                'plugins' => $plugin_list,
                'total' => count($plugin_list)
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "Error listing plugins: " . $e->getMessage()
            ];
        }
    }

    /**
     * Get plugin file path from plugin slug
     *
     * @param string $plugin_slug Plugin slug
     * @return string|false Plugin file path or false if not found
     */
    public function get_plugin_file($plugin_slug)
    {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = get_plugins();

        foreach ($plugins as $plugin_file => $plugin_data) {
            $current_slug = dirname($plugin_file);
            if ($current_slug === $plugin_slug || $plugin_file === $plugin_slug . '.php') {
                return $plugin_file;
            }
        }

        return false;
    }

    /**
     * Check if plugin is installed
     *
     * @param string $plugin_slug Plugin slug
     * @return bool True if installed
     */
    public function is_plugin_installed($plugin_slug)
    {
        $plugin_file = $this->get_plugin_file($plugin_slug);
        return $plugin_file && file_exists(WP_PLUGIN_DIR . '/' . $plugin_file);
    }

    /**
     * Check if plugin is active
     *
     * @param string $plugin_slug Plugin slug
     * @return bool True if active
     */
    public function is_plugin_active($plugin_slug)
    {
        $plugin_file = $this->get_plugin_file($plugin_slug);
        return $plugin_file && is_plugin_active($plugin_file);
    }

    /**
     * Activate a plugin
     *
     * @param string $plugin_slug Plugin slug
     * @return array Operation result
     */
    public function activate_plugin($plugin_slug)
    {
        try {
            $plugin_file = $this->get_plugin_file($plugin_slug);

            if (!$plugin_file) {
                return [
                    'success' => false,
                    'message' => "Plugin '{$plugin_slug}' is not installed",
                    'plugin_slug' => $plugin_slug
                ];
            }

            if (is_plugin_active($plugin_file)) {
                return [
                    'success' => true,
                    'message' => "Plugin '{$plugin_slug}' is already active",
                    'plugin_slug' => $plugin_slug,
                    'status' => 'already_active'
                ];
            }

            $result = activate_plugin($plugin_file);
            if (is_wp_error($result)) {
                return [
                    'success' => false,
                    'message' => "Failed to activate plugin '{$plugin_slug}': " . $result->get_error_message(),
                    'plugin_slug' => $plugin_slug
                ];
            }

            return [
                'success' => true,
                'message' => "Plugin '{$plugin_slug}' activated successfully",
                'plugin_slug' => $plugin_slug,
                'status' => 'activated'
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "Error activating plugin '{$plugin_slug}': " . $e->getMessage(),
                'plugin_slug' => $plugin_slug
            ];
        }
    }

    /**
     * Deactivate a plugin
     *
     * @param string $plugin_slug Plugin slug
     * @return array Operation result
     */
    public function deactivate_plugin($plugin_slug)
    {
        try {
            $plugin_file = $this->get_plugin_file($plugin_slug);

            if (!$plugin_file) {
                return [
                    'success' => false,
                    'message' => "Plugin '{$plugin_slug}' is not installed",
                    'plugin_slug' => $plugin_slug
                ];
            }

            if (!is_plugin_active($plugin_file)) {
                return [
                    'success' => true,
                    'message' => "Plugin '{$plugin_slug}' is already inactive",
                    'plugin_slug' => $plugin_slug,
                    'status' => 'already_inactive'
                ];
            }

            deactivate_plugins($plugin_file);

            return [
                'success' => true,
                'message' => "Plugin '{$plugin_slug}' deactivated successfully",
                'plugin_slug' => $plugin_slug,
                'status' => 'deactivated'
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => "Error deactivating plugin '{$plugin_slug}': " . $e->getMessage(),
                'plugin_slug' => $plugin_slug
            ];
        }
    }
}