<?php
/**
 * Post Manager Class
 *
 * Provides methods for creating, retrieving, updating, and deleting WordPress posts,
 * categories, and tags.
 *
 * @package    WVC_Theme
 * @subpackage Content_Managers
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
    exit;
}

/**
 * Class WVC_Post_Manager
 *
 * Handles operations for WordPress posts, categories, and tags
 */
class WVC_Post_Manager {

    /**
     * Instance of this singleton class
     *
     * @var WVC_Post_Manager
     */
    private static $instance = null;

    /**
     * Get instance of this class
     *
     * @return WVC_Post_Manager
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Private constructor to prevent direct instantiation
    }
    
    /**
     * Upload media to WordPress media library, supporting raw binary data uploads
     *
     * @param mixed $file_data Either raw binary data or file path
     * @param string $filename The filename to use
     * @param string $mime_type The MIME type of the file
     * @param int $post_id Optional post ID to attach the media to
     * @param array $args Optional arguments for media upload
     *
     * @return int|WP_Error Media ID on success, WP_Error on failure
     */
    public function upload_media($file_data, $filename, $mime_type, $post_id = 0, $args = array()) {
        // Include necessary files for media handling
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        
        // Default arguments
        $defaults = array(
            'title'       => '',
            'caption'     => '',
            'description' => '',
            'alt_text'    => '',
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Create temporary file
        $upload_dir = wp_upload_dir();
        $temp_file = $upload_dir['path'] . '/' . uniqid() . '-' . $filename;
        
        // Check if file_data is a string of raw binary data
        if (is_string($file_data) && !file_exists($file_data)) {
            // Write the binary data to the temporary file
            $file_result = file_put_contents($temp_file, $file_data);
            
            if ($file_result === false) {
                return new WP_Error(
                    'file_save_failed',
                    __('Failed to save uploaded file', 'wvc-theme'),
                    array('status' => 500)
                );
            }
        } 
        // Handle cases where file_data is a file path
        elseif (is_string($file_data) && file_exists($file_data)) {
            // Copy the file to our temp location
            if (!copy($file_data, $temp_file)) {
                return new WP_Error(
                    'file_copy_failed',
                    __('Failed to copy source file', 'wvc-theme'),
                    array('status' => 500)
                );
            }
        }
        else {
            return new WP_Error(
                'invalid_file_data',
                __('Invalid file data provided', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Prepare file data for sideload
        $file_array = array(
            'name' => $filename,
            'tmp_name' => $temp_file,
            'type' => $mime_type,
            'error' => 0,
            'size' => filesize($temp_file),
        );
        
        // Process the uploaded file (moves from temp to uploads directory)
        $attachment_id = media_handle_sideload($file_array, $post_id);
        
        // Clean up the temporary file regardless of success
        @unlink($temp_file);
        
        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }
        
        // Update attachment metadata if needed
        if (!empty($args['title'])) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_title' => $args['title'],
            ));
        }
        
        if (!empty($args['caption'])) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_excerpt' => $args['caption'],
            ));
        }
        
        if (!empty($args['description'])) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_content' => $args['description'],
            ));
        }
        
        if (!empty($args['alt_text'])) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $args['alt_text']);
        }
        
        return $this->get_media($attachment_id);
    }
    
    /**
     * Get media item details
     *
     * @param int $media_id Media ID
     *
     * @return array|WP_Error Media data on success, WP_Error on failure
     */
    public function get_media( $media_id ) {
        $attachment = get_post( $media_id );
        
        if ( ! $attachment || 'attachment' !== $attachment->post_type ) {
            return new WP_Error(
                'media_not_found',
                __( 'Media not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }
        
        // Get attachment metadata
        $metadata = wp_get_attachment_metadata( $media_id );
        $attachment_url = wp_get_attachment_url( $media_id );
        
        // Get image sizes if it's an image
        $sizes = array();
        if ( wp_attachment_is_image( $media_id ) && isset( $metadata['sizes'] ) ) {
            foreach ( $metadata['sizes'] as $size => $size_data ) {
                $image_src = wp_get_attachment_image_src( $media_id, $size );
                if ( $image_src ) {
                    $sizes[ $size ] = array(
                        'url'    => $image_src[0],
                        'width'  => $image_src[1],
                        'height' => $image_src[2],
                    );
                }
            }
        }
        
        // Get alt text
        $alt_text = get_post_meta( $media_id, '_wp_attachment_image_alt', true );
        
        $media_data = array(
            'id'          => $attachment->ID,
            'title'       => $attachment->post_title,
            'caption'     => $attachment->post_excerpt,
            'description' => $attachment->post_content,
            'alt_text'    => $alt_text,
            'mime_type'   => $attachment->post_mime_type,
            'url'         => $attachment_url,
            'sizes'       => $sizes,
            'uploaded_on' => $attachment->post_date,
        );
        
        return $media_data;
    }
    
    /**
     * Update media item
     *
     * @param int   $media_id Media ID
     * @param array $args     Arguments for updating the media
     *
     * @return array|WP_Error Updated media data on success, WP_Error on failure
     */
    public function update_media( $media_id, $args ) {
        $attachment = get_post( $media_id );
        
        if ( ! $attachment || 'attachment' !== $attachment->post_type ) {
            return new WP_Error(
                'media_not_found',
                __( 'Media not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }
        
        $update_data = array(
            'ID' => $media_id,
        );
        
        if ( isset( $args['title'] ) ) {
            $update_data['post_title'] = $args['title'];
        }
        
        if ( isset( $args['caption'] ) ) {
            $update_data['post_excerpt'] = $args['caption'];
        }
        
        if ( isset( $args['description'] ) ) {
            $update_data['post_content'] = $args['description'];
        }
        
        // Update the attachment post
        if ( count( $update_data ) > 1 ) {
            $result = wp_update_post( $update_data, true );
            
            if ( is_wp_error( $result ) ) {
                return $result;
            }
        }
        
        // Update alt text if provided
        if ( isset( $args['alt_text'] ) ) {
            update_post_meta( $media_id, '_wp_attachment_image_alt', $args['alt_text'] );
        }
        
        return $this->get_media( $media_id );
    }
    
    /**
     * Delete media item
     *
     * @param int  $media_id Media ID
     * @param bool $force    Whether to force delete or move to trash
     *
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_media( $media_id, $force = false ) {
        $attachment = get_post( $media_id );
        
        if ( ! $attachment || 'attachment' !== $attachment->post_type ) {
            return new WP_Error(
                'media_not_found',
                __( 'Media not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }
        
        $result = wp_delete_attachment( $media_id, $force );
        
        if ( ! $result ) {
            return new WP_Error(
                'delete_failed',
                __( 'Failed to delete media', 'wvc-theme' ),
                array( 'status' => 500 )
            );
        }
        
        return array(
            'deleted'  => true,
            'media_id' => $media_id,
        );
    }

    /**
     * Create a new post
     *
     * @param array $args Arguments for creating a post
     *
     * @return int|WP_Error Post ID on success, WP_Error on failure
     */
    public function create_post( $args ) {
        $post_type = isset( $args['post_type'] ) ? $args['post_type'] : 'post';
        $post_data = array(
			'post_title'     => isset( $args['title'] ) ? $args['title'] : '',
			'post_content'   => isset( $args['content'] ) ? $args['content'] : '',
			'post_status'    => isset( $args['status'] ) ? $args['status'] : 'draft',
			'post_author'    => isset( $args['author'] ) ? $args['author'] : get_current_user_id(),
			'post_excerpt'   => isset( $args['excerpt'] ) ? $args['excerpt'] : '',
			'post_type'      => $post_type,  // DYNAMIC: supports CPTs
			'post_parent'    => isset( $args['parent'] ) ? $args['parent'] : 0,
			'menu_order'     => isset( $args['menu_order'] ) ? $args['menu_order'] : 0,
			'comment_status' => isset( $args['comment_status'] ) ? $args['comment_status'] : 'open',
			'ping_status'    => isset( $args['ping_status'] ) ? $args['ping_status'] : 'open',
        );

        // Insert the post
        $post_id = wp_insert_post( $post_data, true );

        if ( is_wp_error( $post_id ) ) {
            return $post_id;
        }

        // Set categories if provided
        if ( isset( $args['categories'] ) && is_array( $args['categories'] ) ) {
            wp_set_post_categories( $post_id, $args['categories'] );
        }

        // Set tags if provided
        if ( isset( $args['tags'] ) && is_array( $args['tags'] ) ) {
            wp_set_post_tags( $post_id, $args['tags'] );
        }

        // Set featured image if provided
        if ( isset( $args['featured_image'] ) && ! empty( $args['featured_image'] ) ) {
            set_post_thumbnail( $post_id, $args['featured_image'] );
        }

		// Set meta values if provided
		if ( isset( $args['meta'] ) && is_array( $args['meta'] ) ) {
			foreach ( $args['meta'] as $meta_key => $meta_value ) {
				// If meta value is empty, try to auto-assign for "another_custom_post_type" fields
				if ( $meta_value === '' || $meta_value === null ) {
					$maybe_random_id = $this->maybe_get_random_related_post_id( $post_data['post_type'], $meta_key );
					if ( $maybe_random_id !== null ) {
						$meta_value = $maybe_random_id;
					}
				}

				update_post_meta( $post_id, $meta_key, $meta_value );
			}
		}
        if($post_type === 'page'){
            \WVC\Core\Core::set_wvc_editor_meta($post_id, true);
        }
        
        return $this->get_post( $post_id );
    }

    /**
     * Get a post with its metadata
     *
     * @param int $post_id Post ID
     *
     * @return array|WP_Error Post data on success, WP_Error on failure
     */
    public function get_post( $post_id ) {
        $post = get_post( $post_id );

        if ( ! $post ) {
            return new WP_Error(
                'post_not_found',
                __( 'Post not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        $post_data = array(
            'id'             => $post->ID,
            'title'          => $post->post_title,
            'content'        => $post->post_content,
            'excerpt'        => $post->post_excerpt,
            'status'         => $post->post_status,
            'author'         => $post->post_author,
            'date'           => $post->post_date,
            'modified'       => $post->post_modified,
            'comment_status' => $post->comment_status,
            'ping_status'    => $post->ping_status,
            'permalink'      => get_permalink( $post->ID ),
            'categories'     => wp_get_post_categories( $post->ID, array( 'fields' => 'ids' ) ),
            'tags'           => wp_get_post_tags( $post->ID, array( 'fields' => 'ids' ) ),
            'featured_image' => get_post_thumbnail_id( $post->ID ),
        );

        // Get all meta fields
        $meta = get_post_meta( $post->ID );
        if ( $meta ) {
            $formatted_meta = array();
            foreach ( $meta as $key => $values ) {
                $formatted_meta[ $key ] = maybe_unserialize( $values[0] );
            }
            $post_data['meta'] = $formatted_meta;
        }

        return $post_data;
    }

    /**
     * Update an existing post
     *
     * @param int   $post_id Post ID
     * @param array $args    Arguments for updating the post
     *
     * @return array|WP_Error Updated post data on success, WP_Error on failure
     */
    public function update_post( $post_id, $args ) {
        $post = get_post( $post_id );

        if ( ! $post ) {
            return new WP_Error(
                'post_not_found',
                __( 'Post not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        $post_data = array(
            'ID' => $post_id,
        );

        // Update post fields if provided
        if ( isset( $args['title'] ) ) {
            $post_data['post_title'] = $args['title'];
        }

        if ( isset( $args['content'] ) ) {
            $post_data['post_content'] = $args['content'];
        }

        if ( isset( $args['excerpt'] ) ) {
            $post_data['post_excerpt'] = $args['excerpt'];
        }

        if ( isset( $args['status'] ) ) {
            $post_data['post_status'] = $args['status'];
        }

        if ( isset( $args['author'] ) ) {
            $post_data['post_author'] = $args['author'];
        }

        if ( isset( $args['parent'] ) ) {
            $post_data['post_parent'] = $args['parent'];
        }

        if ( isset( $args['menu_order'] ) ) {
            $post_data['menu_order'] = $args['menu_order'];
        }

        if ( isset( $args['comment_status'] ) ) {
            $post_data['comment_status'] = $args['comment_status'];
        }

        if ( isset( $args['ping_status'] ) ) {
            $post_data['ping_status'] = $args['ping_status'];
        }

        // Update the post
        $updated = wp_update_post( $post_data, true );

        if ( is_wp_error( $updated ) ) {
            return $updated;
        }

        // Update categories if provided
        if ( isset( $args['categories'] ) && is_array( $args['categories'] ) ) {
            wp_set_post_categories( $post_id, $args['categories'] );
        }

        // Update tags if provided
        if ( isset( $args['tags'] ) && is_array( $args['tags'] ) ) {
            wp_set_post_tags( $post_id, $args['tags'] );
        }

        // Update featured image if provided
        if ( isset( $args['featured_image'] ) ) {
            if ( empty( $args['featured_image'] ) ) {
                delete_post_thumbnail( $post_id );
            } else {
                set_post_thumbnail( $post_id, $args['featured_image'] );
            }
        }

		// Update meta values if provided
		if ( isset( $args['meta'] ) && is_array( $args['meta'] ) ) {
			foreach ( $args['meta'] as $meta_key => $meta_value ) {
				// If meta value is empty, try to auto-assign for "another_custom_post_type" fields
				if ( $meta_value === '' || $meta_value === null ) {
					$maybe_random_id = $this->maybe_get_random_related_post_id( $post->post_type, $meta_key );
					if ( $maybe_random_id !== null ) {
						$meta_value = $maybe_random_id;
					}
				}

				update_post_meta( $post_id, $meta_key, $meta_value );
			}
		}

        return $this->get_post( $post_id );
    }

    /**
     * Delete a post
     *
     * @param int  $post_id Post ID
     * @param bool $force   Whether to force delete or move to trash
     *
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function delete_post( $post_id, $force = false ) {
        $post = get_post( $post_id );

        if ( ! $post ) {
            return new WP_Error(
                'post_not_found',
                __( 'Post not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        $result = wp_delete_post( $post_id, $force );

        if ( ! $result ) {
            return new WP_Error(
                'delete_failed',
                __( 'Failed to delete post', 'wvc-theme' ),
                array( 'status' => 500 )
            );
        }

        return array(
            'deleted' => true,
            'post_id' => $post_id
        );
    }

    /**
     * Create a new category
     *
     * @param array $args Arguments for creating a category
     *
     * @return array|WP_Error Category data on success, WP_Error on failure
     */
    public function create_category( $args ) {
		return \WVC_Term_Manager::get_instance()->create_term( 'category', $args );
    }

    /**
     * Get a category with its metadata
     *
     * @param int $category_id Category ID
     *
     * @return array|WP_Error Category data on success, WP_Error on failure
     */
    public function get_category( $category_id ) {
		return \WVC_Term_Manager::get_instance()->get_term( $category_id, 'category' );
    }

    /**
     * Update a category
     *
     * @param int   $category_id Category ID
     * @param array $args        Arguments for updating the category
     *
     * @return array|WP_Error Updated category data on success, WP_Error on failure
     */
    public function update_category( $category_id, $args ) {
		return \WVC_Term_Manager::get_instance()->update_term( $category_id, 'category', $args );
    }

    /**
     * Delete a category
     *
     * @param int $category_id Category ID
     *
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_category( $category_id ) {
		return \WVC_Term_Manager::get_instance()->delete_term( $category_id, 'category' );
    }

    /**
     * Create a new tag
     *
     * @param array $args Arguments for creating a tag
     *
     * @return array|WP_Error Tag data on success, WP_Error on failure
     */
    public function create_tag( $args ) {
		return \WVC_Term_Manager::get_instance()->create_term( 'post_tag', $args );
    }

    /**
     * Get a tag with its metadata
     *
     * @param int $tag_id Tag ID
     *
     * @return array|WP_Error Tag data on success, WP_Error on failure
     */
    public function get_tag( $tag_id ) {
		return \WVC_Term_Manager::get_instance()->get_term( $tag_id, 'post_tag' );
    }

    /**
     * Update a tag
     *
     * @param int   $tag_id Tag ID
     * @param array $args   Arguments for updating the tag
     *
     * @return array|WP_Error Updated tag data on success, WP_Error on failure
     */
    public function update_tag( $tag_id, $args ) {
		return \WVC_Term_Manager::get_instance()->update_term( $tag_id, 'post_tag', $args );
    }

    /**
     * Delete a tag
     *
     * @param int $tag_id Tag ID
     *
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_tag( $tag_id ) {
		return \WVC_Term_Manager::get_instance()->delete_term( $tag_id, 'post_tag' );
    }

	/**
	 * If a meta key corresponds to a CPT field of type "another_custom_post_type"
	 * and its value is empty, pick a random related post ID from the target CPT.
	 *
	 * Meta keys for dynamic CPT fields follow the pattern:
	 *   wvc_{post_type}_{field_key}
	 * In this implementation, we assume the field key is also the related CPT key.
	 *
	 * @param string $parent_post_type The post type being created/updated.
	 * @param string $meta_key         The meta key being set.
	 *
	 * @return int|null Random related post ID or null if not applicable.
	 */
	private function maybe_get_random_related_post_id( $parent_post_type, $meta_key ) {
		if ( empty( $parent_post_type ) || empty( $meta_key ) ) {
			return null;
		}

		$prefix = 'wvc_' . $parent_post_type . '_';

		// Only handle meta keys that follow our dynamic CPT meta pattern
		if ( strpos( $meta_key, $prefix ) !== 0 ) {
			return null;
		}

		$field_key = substr( $meta_key, strlen( $prefix ) );
		if ( $field_key === '' ) {
			return null;
		}

		// Load field schema for this CPT
		$fields = get_option( 'wvc_cpt_fields_' . $parent_post_type, array() );
		if ( empty( $fields ) || ! is_array( $fields ) ) {
			return null;
		}

		$field_def = null;
		foreach ( $fields as $field ) {
			if ( isset( $field['key'] ) && $field['key'] === $field_key ) {
				$field_def = $field;
				break;
			}
		}

		if ( ! $field_def || empty( $field_def['type'] ) || $field_def['type'] !== 'another_custom_post_type' ) {
			return null;
		}

		// For now we assume the field key is the related CPT key
		$related_post_type = $field_key;

		if ( ! post_type_exists( $related_post_type ) ) {
			return null;
		}

		$posts = get_posts(
			array(
				'post_type'      => $related_post_type,
				'posts_per_page' => 100,
				'fields'         => 'ids',
			)
		);

		if ( empty( $posts ) || ! is_array( $posts ) ) {
			return null;
		}

		$random_index = array_rand( $posts );

		return isset( $posts[ $random_index ] ) ? (int) $posts[ $random_index ] : null;
	}
}