<?php
/**
 * Product Manager Class
 *
 * Provides methods for creating, retrieving, updating, and deleting WooCommerce products,
 * product categories, and product tags.
 *
 * @package    WVC_Theme
 * @subpackage Content_Managers
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
    exit;
}

/**
 * Class WVC_Product_Manager
 *
 * Handles operations for WooCommerce products, product categories, and product tags
 */
class WVC_Product_Manager {

    /**
     * Instance of this singleton class
     *
     * @var WVC_Product_Manager
     */
    private static $instance = null;

    /**
     * Get instance of this class
     *
     * @return WVC_Product_Manager
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Private constructor to prevent direct instantiation
    }

    /**
     * Create a new WooCommerce product
     *
     * @param array $args Product data
     * @return array|WP_Error Product data on success, WP_Error on failure
     */
    public function create_product( $args ) {
        // Validate required fields (only title and price are required)
        $required_fields = array( "title", "price" );
        foreach ( $required_fields as $field ) {
            if ( ! isset( $args[ $field ] ) || $args[ $field ] === "" ) {
                return new WP_Error(
                    "missing_field",
                    sprintf( __( "Missing required field: %s", "wvc-theme" ), $field ),
                    array( "status" => 400 )
                );
            }
        }

        // Validate price
        if ( ! is_numeric( $args[ "price" ] ) || $args[ "price" ] < 0 ) {
            return new WP_Error(
                "invalid_price",
                __( "Price must be a positive number", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        // Validate discounted price if provided
        if ( isset( $args[ "discounted_price" ] ) && $args[ "discounted_price" ] !== null ) {
            if ( ! is_numeric( $args[ "discounted_price" ] ) || $args[ "discounted_price" ] < 0 ) {
                return new WP_Error(
                    "invalid_discounted_price",
                    __( "Discounted price must be a positive number", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }
        }

        // Validate categories if provided (slugs)
        if ( isset( $args[ "categories" ] ) ) {
            if ( ! is_array( $args[ "categories" ] ) ) {
                return new WP_Error(
                    "invalid_categories",
                    __( "Categories must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "categories" ] as $category_slug ) {
                if ( ! preg_match( "/^[a-z0-9-_]+$/", $category_slug ) ) {
                    return new WP_Error(
                        "invalid_category_slug",
                        sprintf( __( "Invalid category slug format: %s", "wvc-theme" ), $category_slug ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Validate category IDs if provided
        if ( isset( $args[ "product_cat_ids" ] ) ) {
            if ( ! is_array( $args[ "product_cat_ids" ] ) ) {
                return new WP_Error(
                    "invalid_category_ids",
                    __( "Product category IDs must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "product_cat_ids" ] as $category_id ) {
                if ( filter_var( $category_id, FILTER_VALIDATE_INT ) === false || (int) $category_id <= 0 ) {
                    return new WP_Error(
                        "invalid_category_id",
                        __( "Product category IDs must be positive integers", "wvc-theme" ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Validate tags if provided (slugs)
        if ( isset( $args[ "tags" ] ) && is_array( $args[ "tags" ] ) ) {
            foreach ( $args[ "tags" ] as $tag_slug ) {
                if ( ! preg_match( "/^[a-z0-9-_]+$/", $tag_slug ) ) {
                    return new WP_Error(
                        "invalid_tag_slug",
                        sprintf( __( "Invalid tag slug format: %s", "wvc-theme" ), $tag_slug ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Validate tag IDs if provided
        if ( isset( $args[ "product_tag_ids" ] ) ) {
            if ( ! is_array( $args[ "product_tag_ids" ] ) ) {
                return new WP_Error(
                    "invalid_tag_ids",
                    __( "Product tag IDs must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "product_tag_ids" ] as $tag_id ) {
                if ( filter_var( $tag_id, FILTER_VALIDATE_INT ) === false || (int) $tag_id <= 0 ) {
                    return new WP_Error(
                        "invalid_tag_id",
                        __( "Product tag IDs must be positive integers", "wvc-theme" ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Create the product
        $product = new WC_Product_Simple();

        // Set basic product data
        $product->set_name( sanitize_text_field( $args[ "title" ] ) );

        // Prefer explicit long/short description keys when provided, otherwise fall back
        $description         = isset( $args[ "long_description" ] ) && $args[ "long_description" ] !== "" ? $args[ "long_description" ] : ( isset( $args[ "content" ] ) ? $args[ "content" ] : "" );
        $short_description   = isset( $args[ "short_description" ] ) && $args[ "short_description" ] !== "" ? $args[ "short_description" ] : ( isset( $args[ "excerpt" ] ) ? $args[ "excerpt" ] : "" );

        $product->set_description( wp_kses_post( $description ) );
        $product->set_short_description( wp_kses_post( $short_description ) );
        $product->set_regular_price( $args[ "price" ] );

        // Set sale price if provided
        if ( isset( $args[ "discounted_price" ] ) && $args[ "discounted_price" ] !== null ) {
            $product->set_sale_price( $args[ "discounted_price" ] );
        }

        // Set product status
        $product->set_status( isset( $args[ "status" ] ) ? $args[ "status" ] : "publish" );

        // Set product visibility
        $product->set_catalog_visibility( "visible" );

        // Set product as virtual (since we're dealing with digital products)
        $product->set_virtual( true );

        // Save the product first to get an ID
        $product_id = $product->save();

        if ( is_wp_error( $product_id ) ) {
            return $product_id;
        }

		// Save external UID to meta if provided
		if ( isset( $args[ 'uid' ] ) && $args[ 'uid' ] !== '' ) {
			update_post_meta( $product_id, 'wvc_uid', sanitize_text_field( $args['uid'] ) );
		}

        // Handle categories if provided
        $category_slugs = isset( $args[ "categories" ] ) ? $args[ "categories" ] : array();
        $category_ids   = isset( $args[ "product_cat_ids" ] ) ? array_map( "intval", $args[ "product_cat_ids" ] ) : array();
        if ( ! empty( $category_slugs ) || ! empty( $category_ids ) ) {
            $this->set_product_categories( $product_id, $category_slugs, $category_ids );
        }

        // Handle tags
        $tag_slugs = isset( $args[ "tags" ] ) && is_array( $args[ "tags" ] ) ? $args[ "tags" ] : array();
        $tag_ids   = isset( $args[ "product_tag_ids" ] ) ? array_map( "intval", $args[ "product_tag_ids" ] ) : array();
        if ( ! empty( $tag_slugs ) || ! empty( $tag_ids ) ) {
            $this->set_product_tags( $product_id, $tag_slugs, $tag_ids );
        }

        // Handle featured image if provided
        if ( isset( $args[ "featured_image_id" ] ) && $args[ "featured_image_id" ] ) {
            $attachment_id = (int) $args[ "featured_image_id" ];
            // Verify the attachment exists
            $attachment = get_post( $attachment_id );
            if ( $attachment && $attachment->post_type === "attachment" ) {
                set_post_thumbnail( $product_id, $attachment_id );
            }
        }

        // Get the created product data
        return $this->get_product( $product_id );
    }

    /**
     * Get a WooCommerce product
     *
     * @param int $product_id Product ID
     * @return array|WP_Error Product data on success, WP_Error on failure
     */
    public function get_product( $product_id ) {
        $product = wc_get_product( $product_id );

        if ( ! $product ) {
            return new WP_Error(
                "product_not_found",
                __( "Product not found", "wvc-theme" ),
                array( "status" => 404 )
            );
        }

        // Get product categories
        $categories = wp_get_post_terms( $product_id, "product_cat", array( "fields" => "slugs" ) );
        if ( is_wp_error( $categories ) ) {
            $categories = array();
        }

        // Get product tags
        $tags = wp_get_post_terms( $product_id, "product_tag", array( "fields" => "slugs" ) );
        if ( is_wp_error( $tags ) ) {
            $tags = array();
        }

        // Get featured image ID
        $featured_image_id = get_post_thumbnail_id( $product_id );

        return array(
            "id"                  => $product->get_id(),
            "title"               => $product->get_name(),
            "short_description"   => $product->get_short_description(),
            "long_description"    => $product->get_description(),
            "price"               => (float) $product->get_regular_price(),
            "discounted_price"    => $product->get_sale_price() ? (float) $product->get_sale_price() : null,
            "link"                => $product->get_permalink(),
            "categories"          => $categories,
            "tags"                => $tags,
            "featured_image_id"   => $featured_image_id ? (int) $featured_image_id : null,
            "status"              => $product->get_status(),
            "date_created"        => $product->get_date_created()->format( "c" ),
            "date_modified"       => $product->get_date_modified()->format( "c" ),
        );
    }

    /**
     * Update a WooCommerce product
     *
     * @param int $product_id Product ID
     * @param array $args Product data
     * @return array|WP_Error Product data on success, WP_Error on failure
     */
    public function update_product( $product_id, $args ) {
        $product = wc_get_product( $product_id );

        if ( ! $product ) {
            return new WP_Error(
                "product_not_found",
                __( "Product not found", "wvc-theme" ),
                array( "status" => 404 )
            );
        }

        // Update title if provided
        if ( isset( $args[ "title" ] ) ) {
            $product->set_name( sanitize_text_field( $args[ "title" ] ) );
        }

        // Update descriptions if provided (prefer explicit long/short description keys with fallbacks)
        if ( isset( $args[ "short_description" ] ) || isset( $args[ "excerpt" ] ) ) {
            $short_description = isset( $args[ "short_description" ] )
                ? $args[ "short_description" ]
                : ( isset( $args[ "excerpt" ] ) ? $args[ "excerpt" ] : "" );

            $product->set_short_description( wp_kses_post( $short_description ) );
        }

        if ( isset( $args[ "long_description" ] ) || isset( $args[ "content" ] ) ) {
            $description = isset( $args[ "long_description" ] )
                ? $args[ "long_description" ]
                : ( isset( $args[ "content" ] ) ? $args[ "content" ] : "" );

            $product->set_description( wp_kses_post( $description ) );
        }

        // Update price if provided
        if ( isset( $args[ "price" ] ) ) {
            if ( ! is_numeric( $args[ "price" ] ) || $args[ "price" ] < 0 ) {
                return new WP_Error(
                    "invalid_price",
                    __( "Price must be a positive number", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }
            $product->set_regular_price( $args[ "price" ] );
        }

        // Update discounted price if provided
        if ( isset( $args[ "discounted_price" ] ) ) {
            if ( $args[ "discounted_price" ] !== null ) {
                if ( ! is_numeric( $args[ "discounted_price" ] ) || $args[ "discounted_price" ] < 0 ) {
                    return new WP_Error(
                        "invalid_discounted_price",
                        __( "Discounted price must be a positive number", "wvc-theme" ),
                        array( "status" => 400 )
                    );
                }
                $product->set_sale_price( $args[ "discounted_price" ] );
            } else {
                $product->set_sale_price( "" );
            }
        }

        // Update status if provided
        if ( isset( $args[ "status" ] ) ) {
            $product->set_status( $args[ "status" ] );
        }

        // Save the product
        $result = $product->save();

        if ( is_wp_error( $result ) ) {
            return $result;
        }

		// Update external UID if provided
		if ( isset( $args['uid'] ) ) {
			if ( $args['uid'] !== '' && $args['uid'] !== null ) {
				update_post_meta( $product_id, 'wvc_uid', sanitize_text_field( $args['uid'] ) );
			}
		}

        // Validate categories if provided (slugs)
        if ( isset( $args[ "categories" ] ) ) {
            if ( ! is_array( $args[ "categories" ] ) ) {
                return new WP_Error(
                    "invalid_categories",
                    __( "Categories must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "categories" ] as $category_slug ) {
                if ( ! preg_match( "/^[a-z0-9-_]+$/", $category_slug ) ) {
                    return new WP_Error(
                        "invalid_category_slug",
                        sprintf( __( "Invalid category slug format: %s", "wvc-theme" ), $category_slug ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Validate category IDs if provided
        if ( isset( $args[ "product_cat_ids" ] ) ) {
            if ( ! is_array( $args[ "product_cat_ids" ] ) ) {
                return new WP_Error(
                    "invalid_category_ids",
                    __( "Product category IDs must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "product_cat_ids" ] as $category_id ) {
                if ( filter_var( $category_id, FILTER_VALIDATE_INT ) === false || (int) $category_id <= 0 ) {
                    return new WP_Error(
                        "invalid_category_id",
                        __( "Product category IDs must be positive integers", "wvc-theme" ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Apply category updates if provided
        if ( isset( $args[ "categories" ] ) || isset( $args[ "product_cat_ids" ] ) ) {
            $category_slugs = isset( $args[ "categories" ] ) ? $args[ "categories" ] : array();
            $category_ids   = isset( $args[ "product_cat_ids" ] ) ? array_map( "intval", $args[ "product_cat_ids" ] ) : array();
            if ( ! empty( $category_slugs ) || ! empty( $category_ids ) ) {
                $this->set_product_categories( $product_id, $category_slugs, $category_ids );
            }
        }

        // Validate tags if provided (slugs)
        if ( isset( $args[ "tags" ] ) ) {
            if ( ! is_array( $args[ "tags" ] ) ) {
                return new WP_Error(
                    "invalid_tags",
                    __( "Tags must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "tags" ] as $tag_slug ) {
                if ( ! preg_match( "/^[a-z0-9-_]+$/", $tag_slug ) ) {
                    return new WP_Error(
                        "invalid_tag_slug",
                        sprintf( __( "Invalid tag slug format: %s", "wvc-theme" ), $tag_slug ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Validate tag IDs if provided
        if ( isset( $args[ "product_tag_ids" ] ) ) {
            if ( ! is_array( $args[ "product_tag_ids" ] ) ) {
                return new WP_Error(
                    "invalid_tag_ids",
                    __( "Product tag IDs must be an array", "wvc-theme" ),
                    array( "status" => 400 )
                );
            }

            foreach ( $args[ "product_tag_ids" ] as $tag_id ) {
                if ( filter_var( $tag_id, FILTER_VALIDATE_INT ) === false || (int) $tag_id <= 0 ) {
                    return new WP_Error(
                        "invalid_tag_id",
                        __( "Product tag IDs must be positive integers", "wvc-theme" ),
                        array( "status" => 400 )
                    );
                }
            }
        }

        // Apply tag updates if provided
        if ( isset( $args[ "tags" ] ) || isset( $args[ "product_tag_ids" ] ) ) {
            $tag_slugs = isset( $args[ "tags" ] ) ? $args[ "tags" ] : array();
            $tag_ids   = isset( $args[ "product_tag_ids" ] ) ? array_map( "intval", $args[ "product_tag_ids" ] ) : array();
            if ( ! empty( $tag_slugs ) || ! empty( $tag_ids ) ) {
                $this->set_product_tags( $product_id, $tag_slugs, $tag_ids );
            }
        }

        // Update featured image if provided
        if ( isset( $args[ "featured_image_id" ] ) ) {
            if ( $args[ "featured_image_id" ] ) {
                $attachment_id = (int) $args[ "featured_image_id" ];
                // Verify the attachment exists
                $attachment = get_post( $attachment_id );
                if ( $attachment && $attachment->post_type === "attachment" ) {
                    set_post_thumbnail( $product_id, $attachment_id );
                }
            } else {
                // Remove featured image if null or empty
                delete_post_thumbnail( $product_id );
            }
        }

        return $this->get_product( $product_id );
    }

    /**
     * Delete a WooCommerce product
     *
     * @param int $product_id Product ID
     * @param bool $force Whether to force delete
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_product( $product_id, $force = false ) {
        $product = wc_get_product( $product_id );

        if ( ! $product ) {
            return new WP_Error(
                "product_not_found",
                __( "Product not found", "wvc-theme" ),
                array( "status" => 404 )
            );
        }

        if ( $force ) {
            $result = wp_delete_post( $product_id, true );
        } else {
            $result = wp_trash_post( $product_id );
        }

        if ( ! $result ) {
            return new WP_Error(
                "delete_failed",
                __( "Failed to delete product", "wvc-theme" ),
                array( "status" => 500 )
            );
        }

        return array(
            "message" => $force ? __( "Product permanently deleted", "wvc-theme" ) : __( "Product moved to trash", "wvc-theme" ),
            "id"      => $product_id,
        );
    }

    /**
     * Set product categories by slugs and IDs
     *
     * @param int $product_id Product ID
     * @param array $category_slugs Array of category slugs
     * @param array $category_ids Array of category IDs
     * @return void
     */
    private function set_product_categories( $product_id, $category_slugs = array(), $category_ids = array() ) {
        $resolved_ids = array();

        foreach ( $category_ids as $raw_id ) {
            $candidate_id = (int) $raw_id;
            if ( $candidate_id <= 0 ) {
                continue;
            }

            $term = get_term( $candidate_id, "product_cat" );
            if ( $term && ! is_wp_error( $term ) ) {
                $resolved_ids[] = $term->term_id;
            }
        }

        foreach ( $category_slugs as $slug ) {
            $slug = (string) $slug;
            if ( $slug === "" ) {
                continue;
            }

            $term = get_term_by( "slug", $slug, "product_cat" );
            if ( $term && ! is_wp_error( $term ) ) {
                $resolved_ids[] = $term->term_id;
            } else {
                // Create category if it doesn't exist
                $term = wp_insert_term( $slug, "product_cat", array( "slug" => $slug ) );
                if ( ! is_wp_error( $term ) ) {
                    $resolved_ids[] = $term["term_id"];
                }
            }
        }

        if ( ! empty( $resolved_ids ) ) {
            $resolved_ids = array_map( "intval", array_unique( $resolved_ids ) );
            wp_set_post_terms( $product_id, $resolved_ids, "product_cat" );
        }
    }

    /**
     * Set product tags by slugs and IDs
     *
     * @param int $product_id Product ID
     * @param array $tag_slugs Array of tag slugs
     * @param array $tag_ids Array of tag IDs
     * @return void
     */
    private function set_product_tags( $product_id, $tag_slugs = array(), $tag_ids = array() ) {
        $resolved_ids = array();

        foreach ( $tag_ids as $raw_id ) {
            $candidate_id = (int) $raw_id;
            if ( $candidate_id <= 0 ) {
                continue;
            }

            $term = get_term( $candidate_id, "product_tag" );
            if ( $term && ! is_wp_error( $term ) ) {
                $resolved_ids[] = $term->term_id;
            }
        }

        foreach ( $tag_slugs as $slug ) {
            $slug = (string) $slug;
            if ( $slug === "" ) {
                continue;
            }

            $term = get_term_by( "slug", $slug, "product_tag" );
            if ( $term && ! is_wp_error( $term ) ) {
                $resolved_ids[] = $term->term_id;
            } else {
                // Create tag if it doesn't exist
                $term = wp_insert_term( $slug, "product_tag", array( "slug" => $slug ) );
                if ( ! is_wp_error( $term ) ) {
                    $resolved_ids[] = $term["term_id"];
                }
            }
        }

        if ( ! empty( $resolved_ids ) ) {
            $resolved_ids = array_map( "intval", array_unique( $resolved_ids ) );
            wp_set_post_terms( $product_id, $resolved_ids, "product_tag" );
        }
    }

    /**
     * Create a new product category
     *
     * @param array $args Category data
     * @return array|WP_Error Category data on success, WP_Error on failure
     */
    public function create_category( $args ) {
		return \WVC_Term_Manager::get_instance()->create_term( "product_cat", $args );
    }

    /**
     * Get a product category
     *
     * @param int $category_id Category ID
     * @return array|WP_Error Category data on success, WP_Error on failure
     */
    public function get_category( $category_id ) {
		return \WVC_Term_Manager::get_instance()->get_term( $category_id, "product_cat" );
    }

    /**
     * Update a product category
     *
     * @param int $category_id Category ID
     * @param array $args Category data
     * @return array|WP_Error Category data on success, WP_Error on failure
     */
    public function update_category( $category_id, $args ) {
		return \WVC_Term_Manager::get_instance()->update_term( $category_id, "product_cat", $args );
    }

    /**
     * Delete a product category
     *
     * @param int $category_id Category ID
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_category( $category_id ) {
		return \WVC_Term_Manager::get_instance()->delete_term( $category_id, "product_cat" );
    }

    /**
     * Create a new product tag
     *
     * @param array $args Tag data
     * @return array|WP_Error Tag data on success, WP_Error on failure
     */
    public function create_tag( $args ) {
		return \WVC_Term_Manager::get_instance()->create_term( "product_tag", $args );
    }

    /**
     * Get a product tag
     *
     * @param int $tag_id Tag ID
     * @return array|WP_Error Tag data on success, WP_Error on failure
     */
    public function get_tag( $tag_id ) {
		return \WVC_Term_Manager::get_instance()->get_term( $tag_id, "product_tag" );
    }

    /**
     * Update a product tag
     *
     * @param int $tag_id Tag ID
     * @param array $args Tag data
     * @return array|WP_Error Tag data on success, WP_Error on failure
     */
    public function update_tag( $tag_id, $args ) {
		return \WVC_Term_Manager::get_instance()->update_term( $tag_id, "product_tag", $args );
    }

    /**
     * Delete a product tag
     *
     * @param int $tag_id Tag ID
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_tag( $tag_id ) {
		return \WVC_Term_Manager::get_instance()->delete_term( $tag_id, "product_tag" );
    }

    /**
     * Upload media to WordPress media library, supporting raw binary data uploads
     *
     * @param mixed $file_data Either raw binary data or file path
     * @param string $filename The filename to use
     * @param string $mime_type The MIME type of the file
     * @param int $post_id Optional post ID to attach the media to
     * @param array $args Optional arguments for media upload
     *
     * @return int|WP_Error Media ID on success, WP_Error on failure
     */
    public function upload_media($file_data, $filename, $mime_type, $post_id = 0, $args = array()) {
        // Create a temporary file
        $upload_dir = wp_upload_dir();
        $temp_file = $upload_dir['path'] . '/' . $filename;
        
        // Write the binary data to the temporary file
        $result = file_put_contents($temp_file, $file_data);
        
        if ($result === false) {
            return new WP_Error(
                'upload_failed',
                __('Failed to write file to upload directory', 'wvc-theme'),
                array('status' => 500)
            );
        }
        
        // Prepare the file array for wp_handle_upload
        $file_array = array(
            'name' => $filename,
            'tmp_name' => $temp_file,
            'type' => $mime_type,
            'error' => 0,
            'size' => filesize($temp_file)
        );
        
        // Handle the upload
        $upload_result = wp_handle_upload($file_array, array('test_form' => false));
        
        if (isset($upload_result['error'])) {
            // Clean up temp file
            unlink($temp_file);
            return new WP_Error(
                'upload_error',
                $upload_result['error'],
                array('status' => 400)
            );
        }
        
        // Create attachment post
        $attachment = array(
            'post_mime_type' => $mime_type,
            'post_title' => isset($args['title']) ? $args['title'] : preg_replace('/\.[^.]+$/', '', $filename),
            'post_content' => isset($args['description']) ? $args['description'] : '',
            'post_excerpt' => isset($args['caption']) ? $args['caption'] : '',
            'post_status' => 'inherit',
            'post_parent' => $post_id
        );
        
        $attachment_id = wp_insert_attachment($attachment, $upload_result['file']);
        
        if (is_wp_error($attachment_id)) {
            // Clean up uploaded file
            unlink($upload_result['file']);
            return $attachment_id;
        }
        
        // Generate attachment metadata
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $attachment_data = wp_generate_attachment_metadata($attachment_id, $upload_result['file']);
        wp_update_attachment_metadata($attachment_id, $attachment_data);
        
        // Set alt text if provided
        if (isset($args['alt_text'])) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($args['alt_text']));
        }
        
        // Clean up temp file
        unlink($temp_file);
        
        return $attachment_id;
    }

    /**
     * Get media information
     *
     * @param int $media_id Media ID
     * @return array|WP_Error Media data on success, WP_Error on failure
     */
    public function get_media($media_id) {
        $attachment = get_post($media_id);
        
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error(
                'media_not_found',
                __('Media not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        $metadata = wp_get_attachment_metadata($media_id);
        $alt_text = get_post_meta($media_id, '_wp_attachment_image_alt', true);
        
        return array(
            'id' => $media_id,
            'title' => $attachment->post_title,
            'caption' => $attachment->post_excerpt,
            'description' => $attachment->post_content,
            'alt_text' => $alt_text,
            'mime_type' => $attachment->post_mime_type,
            'url' => wp_get_attachment_url($media_id),
            'sizes' => isset($metadata['sizes']) ? $metadata['sizes'] : array(),
            'date_created' => $attachment->post_date,
            'date_modified' => $attachment->post_modified,
        );
    }

    /**
     * Update media information
     *
     * @param int $media_id Media ID
     * @param array $args Media data
     * @return array|WP_Error Media data on success, WP_Error on failure
     */
    public function update_media($media_id, $args) {
        $attachment = get_post($media_id);
        
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error(
                'media_not_found',
                __('Media not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        $update_args = array('ID' => $media_id);
        
        if (isset($args['title'])) {
            $update_args['post_title'] = sanitize_text_field($args['title']);
        }
        
        if (isset($args['caption'])) {
            $update_args['post_excerpt'] = sanitize_textarea_field($args['caption']);
        }
        
        if (isset($args['description'])) {
            $update_args['post_content'] = sanitize_textarea_field($args['description']);
        }
        
        $result = wp_update_post($update_args);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        if (isset($args['alt_text'])) {
            update_post_meta($media_id, '_wp_attachment_image_alt', sanitize_text_field($args['alt_text']));
        }
        
        return $this->get_media($media_id);
    }

    /**
     * Delete media
     *
     * @param int $media_id Media ID
     * @param bool $force Whether to force delete
     * @return array|WP_Error Success message on success, WP_Error on failure
     */
    public function delete_media($media_id, $force = false) {
        $attachment = get_post($media_id);
        
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error(
                'media_not_found',
                __('Media not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        if ($force) {
            $result = wp_delete_attachment($media_id, true);
        } else {
            $result = wp_trash_post($media_id);
        }
        
        if (!$result) {
            return new WP_Error(
                'delete_failed',
                __('Failed to delete media', 'wvc-theme'),
                array('status' => 500)
            );
        }
        
        return array(
            'message' => $force ? __('Media permanently deleted', 'wvc-theme') : __('Media moved to trash', 'wvc-theme'),
            'id' => $media_id,
        );
    }
}
