<?php

namespace WVC\Editor;

/**
 * Editor
 *
 * @package WVC_Theme
 * @subpackage Editor
 * @author 10Web
 * @since 1.0.0
 * @version 1.0.0
 */

// Prevent direct access
if (! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Editor Class
 * 
 * Handles custom editor functionality for pages
 */
class Editor
{



    /**
     * Post object
     */
    private $post;

    /**
     * Main post ID
     */
    private $main_id;

    /**
     * Constructor
     * 
     * @param \WP_Post $post Post object
     */
    public function __construct($post = null)
    {
        if ($post) {
            $this->post = $post;
        }

        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Hook into load-post.php to handle the custom action
        add_action('load-post.php', array($this, "handle_wvc_editor_action"));

        // Hook into admin_init to handle wvc-editor page early


        // Register hidden admin page for permissions
        add_action('admin_menu', function () {
            // Add hidden admin page (as a submenu under null parent)
            add_submenu_page(
                null,                        // parent slug = null → no menu entry
                'WVC Editor',                // page title
                'WVC Editor',                // menu title (not shown)
                'edit_pages',                // capability
                'wvc-editor',                // slug
                array($this, 'handle_wvc_editor_page') // callback
            );
        });

        // Intercept wvc-editor page requests early
        add_action('admin_init', function () {
            if (isset($_GET['page']) && $_GET['page'] === 'wvc-editor') {
                // Handle the request immediately and exit
                $this->handle_wvc_editor_page();
            }
        });

        // Remove WordPress core styles when WVC editor is active (admin only)
        add_action('admin_enqueue_scripts', array($this, "remove_core_styles"), 1);
    }



    /**
     * Enqueue scripts and styles for the WVC editor
     */
    public function enqueue_editor_assets()
    {
        // Handle case when there is no post - just keep post id null
        $post_id = $this->post ? $this->post->ID : null;
        $term_id = null;
        $taxonomy = null;

        wp_enqueue_media();

        wp_enqueue_script("wvc-js-utils", get_template_directory_uri() . "/assets/js/index.umd.js", array(), wp_get_theme()->get('Version'), true);

        wp_enqueue_script('wvc', get_template_directory_uri() . '/assets/js/wvc.js', array('wvc-js-utils'), null, false);

        // Enqueue external CSS
        wp_enqueue_style(
            'wvc-editor-wrapper',
            get_template_directory_uri() . '/assets/css/wvc-editor.css',
            array(),
            wp_get_theme()->get('Version'),
        );
        wp_enqueue_style(
            'wvc-editor-styles',
            get_template_directory_uri() . '/app/assets/index.css',
            array(),
            wp_get_theme()->get('Version'),
        );





        // Enqueue external JavaScript
        wp_enqueue_script(
            'wvc-editor',
            get_template_directory_uri() . '/app/assets/index.js',
            array("wvc-js-utils", "wvc"),
            wp_get_theme()->get('Version'),
        );


        # important - we actulally need to wp_enqueue_script_module but it's not working because of the problems with dependencies and localized varaiables
        // Localize script with all necessary variables under WVC object
        wp_localize_script('wvc', 'WVC', \WVC\Core\Core::get_wvc_params_to_localize($post_id, $term_id, $taxonomy, true));

        // Debug: Check if localization worked
        error_log('WVC Editor: Script localized with WVC');
    }

    /**
     * Check if current page is a WVC editor page
     * 
     * @return bool True if current page is WVC editor page
     */
    public function is_editor_page()
    {
        return self::is_editor_page_static();
    }

    /**
     * Check if current page is a WVC editor page (static version)
     * 
     * @return bool True if current page is WVC editor page
     */
    public static function is_editor_page_static()
    {
        global $pagenow;

        // Check if we're on the correct admin page
        if ($pagenow !== 'post.php' && $pagenow !== 'admin.php') {
            return false;
        }

        // Check if we're on a WVC editor action (post.php route)
        if ($pagenow === 'post.php' && (!isset($_GET['action']) || $_GET['action'] !== 'wvc-editor')) {
            return false;
        }

        // Check if we're on the wvc-editor page (admin.php route)
        if ($pagenow === 'admin.php' && (!isset($_GET['page']) || $_GET['page'] !== 'wvc-editor')) {
            return false;
        }

        return true;
    }

    /**
     * Remove WordPress core styles and scripts when WVC editor is active
     */
    public function remove_core_styles()
    {
        // Check if we're on a WVC editor page
        if (!$this->is_editor_page()) {
            return;
        }

        // Remove WordPress core duotone styles
        wp_dequeue_style('wp-block-library');
        wp_deregister_style('wp-block-library');

        // Remove duotone styles specifically
        remove_action('wp_enqueue_scripts', array('WP_Duotone', 'output_block_styles'), 9);
        remove_action('wp_enqueue_scripts', array('WP_Duotone', 'output_global_styles'), 11);
        remove_action('admin_enqueue_scripts', array('WP_Duotone', 'output_block_styles'), 9);
        remove_action('admin_enqueue_scripts', array('WP_Duotone', 'output_global_styles'), 11);

        // Remove other core styles that might interfere
        wp_dequeue_style('wp-admin');
        wp_dequeue_style('dashicons');
        wp_dequeue_style('admin-bar');

        // Remove block editor, Elementor and other template scripts
        $this->remove_other_scripts();
    }

    /**
     * Remove template scripts that might interfere with WVC editor
     */
    private function remove_other_scripts()
    {
        global $wp_scripts;

        if (!$wp_scripts) {
            return;
        }

        // Get all registered scripts
        $registered_scripts = $wp_scripts->registered;

        // Remove scripts with specific patterns
        $patterns_to_remove = array(
            'tmpl-elementor-',  // Elementor templates
            'tmpl-',            // Any template scripts
            'elementor-',       // Elementor scripts
            'wp-block-',        // WordPress block scripts
            'wp-editor-',       // WordPress editor scripts
            'wp-components-',   // WordPress components
            'wp-data-',         // WordPress data
            'wp-element-',      // WordPress element
            'wp-i18n-',         // WordPress internationalization
            'wp-polyfill-',     // WordPress polyfills
            'wp-url-',          // WordPress URL utilities
            'wp-hooks-',        // WordPress hooks
            'wp-deprecated-',   // WordPress deprecated
            'wp-api-fetch-',    // WordPress API fetch
            'wp-blob-',         // WordPress blob
            'wp-core-data-',    // WordPress core data
            'wp-edit-post-',    // WordPress edit post
            'wp-edit-site-',    // WordPress edit site
            # 'wp-plugins-',      // WordPress plugins
            'wp-priority-queue-', // WordPress priority queue
            'wp-redux-routine-',  // WordPress redux routine
            'wp-reusable-blocks-', // WordPress reusable blocks
            'wp-server-side-render-', // WordPress server side render
            # 'wp-shortcode-',    // WordPress shortcode
            'wp-token-list-',   // WordPress token list
            'wp-viewport-',     // WordPress viewport
            'wp-wordcount-',    // WordPress wordcount
        );

        foreach ($registered_scripts as $handle => $script) {
            // Skip our own script
            if ($handle === 'wvc-editor') {
                continue;
            }

            foreach ($patterns_to_remove as $pattern) {
                if (strpos($handle, $pattern) === 0) {
                    wp_dequeue_script($handle);
                    wp_deregister_script($handle);
                    break;
                }
            }
        }
    }







    /**
     * Handle custom wvc-editor action
     * 
     * Intercepts the wvc-editor action and loads the custom editor view
     */
    public function handle_wvc_editor_action()
    {
        // Check if we're on a WVC editor page (post.php route)
        if (!$this->is_editor_page() || !isset($_GET['action'])) {
            return;
        }

        // Check if post ID is provided
        if (!isset($_GET['post']) || empty($_GET['post'])) {
            wp_die(__('No post ID provided.', 'wvc-theme'));
        }

        $post_id = intval($_GET['post']);
        $post = get_post($post_id);

        // need to check later if we have single post or single product templates to edit
        // Verify the post exists and is a page
        if (!$post || ($post->post_type !== 'page' && $post->post_type !== 'post' && $post->post_type !== 'product')) {
            wp_die(__('Invalid post or not a page.', 'wvc-theme'));
        }

        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            wp_die(__('You do not have permission to edit this page.', 'wvc-theme'));
        }

        $is_built_with_wvc = \WVC\Core\Core::is_built_with_wvc($post_id);

        if (!$is_built_with_wvc && $post->post_type !== 'post' && $post->post_type !== 'product') {
            return;
        }

        // Set the post for this instance
        $this->set_post($post);

        // Enqueue editor assets before loading the view
        $this->enqueue_editor_assets();

        // Load the custom editor view
        $this->load_editor_view($post);
        exit;
    }

    /**
     * Handle the wvc-editor page request
     */
    public function handle_wvc_editor_page()
    {
        // Check if post ID is provided
        if (!isset($_GET['post']) || empty($_GET['post'])) {
            // Auto-select a page based on priority
            $post_id = $this->get_default_page_id();
            if ($post_id) {
                // Redirect to the selected page
                $redirect_url = add_query_arg('post', $post_id);
                wp_redirect($redirect_url);
                exit;
            } else {
                // No WVC pages exist, open editor without post ID
                $this->load_editor_without_post();
                exit;
            }
        }

        $post_id = intval($_GET['post']);
        $post = get_post($post_id);

        // Verify the post exists and is a page
        if (!$post || ($post->post_type !== 'page' && $post->post_type !== 'post')) {
            wp_die(__('Invalid post or not a page.', 'wvc-theme'));
        }

        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            wp_die(__('You do not have permission to edit this page.', 'wvc-theme'));
        }

        $is_built_with_wvc = \WVC\Core\Core::is_built_with_wvc($post_id);

        if (!$is_built_with_wvc && $post->post_type !== 'post') {
            return;
        }

        // Set the post for this instance
        $this->set_post($post);

        // Enqueue editor assets before loading the view
        $this->enqueue_editor_assets();

        // Load the custom editor view and exit immediately
        $this->load_editor_view($post);
        exit;
    }

    /**
     * Get the default page ID based on priority
     * 
     * @return int|false Page ID or false if no WVC pages found
     */
    public function get_default_page_id()
    {
        $front_pages = get_option('show_on_front');
        // First, check if homepage is set to a page and that page is built with WVC
        $front_page_id = get_option('page_on_front');
        if ($front_pages === 'page' && $front_page_id && \WVC\Core\Core::is_built_with_wvc($front_page_id)) {
            return $front_page_id;
        }

        // Otherwise, get a WVC page prioritizing _wvc_home=true, then lowest ID
        $wvc_pages = get_posts(array(
            'post_type' => 'page',
            'post_status'    => 'any',
            'posts_per_page' => 1,
            'orderby'        => array('ID' => 'ASC'),
            'meta_query'     => [
                'relation' => 'AND',
                
                [
                    'key'     => '_wvc_home',
                    'value' => '1',
                    'compare' => '='
                ],
                [
                    'key'     => \WVC\Core\Core::WVC_EDITOR_META_KEY,
                    'value'   => '1',
                    'compare' => '='
                ],
            ],
            'fields'         => 'ids',
        ));

        if (!empty($wvc_pages)) {
            return $wvc_pages[0];
        }

        // Otherwise, get the WVC page with the lowest post ID
        $wvc_pages = get_posts(array(
            'post_type' => 'page',
            'post_status'    => 'any',
            'posts_per_page' => 1,
            'orderby' => 'ID',
            'order' => 'ASC',
            'meta_query'     => [
                [
                    'key'     => \WVC\Core\Core::WVC_EDITOR_META_KEY,
                    'value'   => '1',
                    'compare' => '='
                ],
            ],
            'fields'         => 'ids',
        ));

        if (!empty($wvc_pages)) {
            return $wvc_pages[0];
        }

        // If no WVC pages exist, return false to open editor without post ID
        return false;
    }

    /**
     * Load the editor without a specific post
     */
    public function load_editor_without_post()
    {
        // Enqueue editor assets
        $this->enqueue_editor_assets();

        // Load the custom editor view without a post
        $this->load_editor_view(null);
        exit;
    }







    /**
     * Get page data for the editor
     * 
     * @return array Page data
     */
    public function get_page_data()
    {
        if (!$this->post) {
            return array();
        }

        return array(
            'id' => $this->post->ID,
            'title' => $this->post->post_title,
            'status' => $this->post->post_status,
            'wvc_editor' => \WVC\Core\Core::get_wvc_editor_meta($this->post->ID),
        );
    }

    /**
     * Get body classes for the editor
     * 
     * @return array Body classes
     */
    public function get_body_classes()
    {
        $classes = array('wvc-editor-body', 'wvc-editor-active');

        if (!$this->post) {
            return $classes;
        }

        $is_built_with_wvc = \WVC\Core\Core::is_built_with_wvc($this->post->ID);

        if ($is_built_with_wvc) {
            $classes[] = 'wvc-editor-built';
        } else {
            $classes[] = 'wvc-editor-not-built';
        }

        return $classes;
    }

    /**
     * Load the custom editor view
     * 
     * @param \WP_Post $post The post object to edit
     */
    public function load_editor_view($post)
    {
        // Set the post for this instance
        $this->set_post($post);

        // Start output buffering to capture any unwanted output
        ob_start();

        // Include the wrapper template
        include get_template_directory() . '/includes/editor/editor-wrapper.php';

        // Get the output and clean it
        $output = ob_get_clean();

        // Remove any unwanted WordPress admin elements that might have been output
        $output = preg_replace('/<div[^>]*id="adminmenumain"[^>]*>.*?<\/div>/s', '', $output);
        $output = preg_replace('/<div[^>]*id="adminmenuwrap"[^>]*>.*?<\/div>/s', '', $output);
        $output = preg_replace('/<div[^>]*id="adminmenu"[^>]*>.*?<\/div>/s', '', $output);

        // Output the clean HTML
        echo $output;
        exit;
    }





    /**
     * Get edit URL for WVC editor
     * 
     * @return string Edit URL
     */
    public function get_edit_url()
    {
        if (!$this->post) {
            return '';
        }

        $url = add_query_arg(
            array(
                'post' => $this->get_main_id(),
                'action' => 'wvc-editor',
            ),
            admin_url('post.php')
        );

        /**
         * WVC Document edit url.
         *
         * Filters the document edit url.
         *
         * @since 1.0.0
         *
         * @param string $url The edit url.
         * @param Editor $this The editor instance.
         */
        $url = apply_filters('wvc/document/urls/edit', $url, $this);

        return $url;
    }

    /**
     * Get edit URL for WVC editor using index.php route
     * 
     * @return string Edit URL
     */
    public function get_index_edit_url()
    {
        if (!$this->post) {
            return '';
        }

        $url = add_query_arg(
            array(
                'page' => 'wvc-editor',
                'post' => $this->get_main_id(),
            ),
            admin_url('index.php')
        );

        /**
         * WVC Document edit url (index.php route).
         *
         * Filters the document edit url for the index.php route.
         *
         * @since 1.0.0
         *
         * @param string $url The edit url.
         * @param Editor $this The editor instance.
         */
        $url = apply_filters('wvc/document/urls/edit_index', $url, $this);

        return $url;
    }

    /**
     * Get main post ID
     * 
     * @return int Post ID
     */
    public function get_main_id()
    {
        if (!$this->post) {
            return 0;
        }

        if (!$this->main_id) {
            $post_id = $this->post->ID;

            $parent_post_id = wp_is_post_revision($post_id);

            if ($parent_post_id) {
                $post_id = $parent_post_id;
            }

            $this->main_id = $post_id;
        }

        return $this->main_id;
    }

    /**
     * Check if current user can edit this post
     * 
     * @return bool
     */
    public function is_editable_by_current_user()
    {
        if (!$this->post) {
            return false;
        }

        return current_user_can('edit_post', $this->get_main_id());
    }

    /**
     * Set post object
     * 
     * @param \WP_Post $post Post object
     * @return $this
     */
    public function set_post($post)
    {
        $this->post = $post;
        $this->main_id = null; // Reset main_id when post changes
        return $this;
    }

    /**
     * Get post object
     * 
     * @return \WP_Post|null
     */
    public function get_post()
    {
        return $this->post;
    }
}
