<?php

namespace WVC\FormSubmissions;

/**
 * Admin List Table - Columns & Filters for Submissions List
 *
 * Manages the submissions list table appearance and filtering
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

/**
 * Add custom columns to the form data admin list
 */
function wvc_add_form_data_columns($columns) {
    $new_columns = array();
    $new_columns['cb'] = $columns['cb'];
    $new_columns['title'] = $columns['title'];
    $new_columns['form_id'] = 'Form ID';
    $new_columns['section_name'] = 'Section';
    $new_columns['submission_date'] = 'Submitted';
    $new_columns['form_fields'] = 'Form Fields';

    return $new_columns;
}

/**
 * Display custom column content
 */
function wvc_display_form_data_columns($column, $post_id) {
    switch ($column) {
        case 'form_id':
            echo esc_html(get_post_meta($post_id, '_wvc_form_id', true));
            break;
        case 'section_name':
            echo esc_html(get_post_meta($post_id, '_wvc_section_name', true));
            break;
        case 'submission_date':
            $timestamp = get_post_meta($post_id, '_wvc_timestamp', true);
            if ($timestamp) {
                echo esc_html(date('Y-m-d H:i:s', $timestamp / 1000));
            }
            break;
        case 'form_fields':
            $form_data_json = get_post_meta($post_id, '_wvc_form_data', true);
            $form_data = $form_data_json ? json_decode($form_data_json, true) : array();
            if (!empty($form_data)) {
                echo esc_html(implode(', ', array_keys($form_data)));
            }
            break;
    }
}

/**
 * Add form ID filter to the form submissions list
 */
function wvc_add_form_id_filter() {
    global $typenow;

    if ($typenow !== 'wvc_form_data') {
        return;
    }

    // Get all registered forms (not just those with submissions)
    $all_forms = wvc_get_all_registered_forms_with_counts();
    $selected_form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';

    echo '<select name="form_id" id="form_id_filter">';
    echo '<option value="">All Forms</option>';

    foreach ($all_forms as $form_stat) {
        $form_key = esc_attr($form_stat['form_key']);
        $form_label = esc_html($form_stat['form_label']);
        $count = $form_stat['submission_count'];
        $selected = selected($selected_form_id, $form_key, false);

        echo '<option value="' . $form_key . '"' . $selected . '>' . $form_label . ' (' . $count . ')</option>';
    }

    echo '</select>';
}

/**
 * Filter form submissions by form ID
 */
function wvc_filter_submissions_by_form_id($query) {
    global $pagenow, $typenow;

    if ($pagenow === 'edit.php' && $typenow === 'wvc_form_data' && isset($_GET['form_id']) && !empty($_GET['form_id'])) {
        $form_id = sanitize_text_field($_GET['form_id']);

        $query->set('meta_key', '_wvc_form_id');
        $query->set('meta_value', $form_id);
    }
}
