<?php

namespace WVC\FormSubmissions;

/**
 * WVC Form Submissions Handler
 *
 * Handles AJAX form submissions and data processing
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.16
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * Handle form submission via AJAX
 */
function wvc_handle_ajax_form_submission() {
    // Verify nonce for security
    if (!check_ajax_referer('wvc_nonce', 'nonce', false)) {
        error_log("Invalid nonce in AJAX form submission");
        wp_send_json_error(array(
            'message' => 'Invalid security token'
        ), 403);
        return;
    }
    
    // Get parameters from $_POST
    $section_name = sanitize_text_field($_POST['sectionName'] ?? '');
    $form_id = sanitize_text_field($_POST['formId'] ?? '');
    $timestamp = floatval($_POST['timestamp'] ?? 0);
    $session_id = sanitize_text_field($_POST['sessionId'] ?? '');
    $page_url = esc_url_raw($_POST['pageUrl'] ?? '');
    $user_agent = sanitize_text_field($_POST['userAgent'] ?? '');
    $submission_attempt = intval($_POST['submissionAttempt'] ?? 1);
    $form_version = sanitize_text_field($_POST['formVersion'] ?? '1.0.0');
    $validation_errors = $_POST['validationErrors'] ?? '';
    
    // Validate required fields
    if (empty($section_name) || empty($form_id)) {
        wp_send_json_error(array(
            'message' => 'Missing required fields: sectionName and formId'
        ), 400);
        return;
    }
    
    // Extract and sanitize form data
    $form_data_raw = array();
    
    // Check if formData comes as an array
    if (isset($_POST['formData']) && is_array($_POST['formData'])) {
        foreach ($_POST['formData'] as $key => $value) {
            $form_data_raw[sanitize_key($key)] = sanitize_textarea_field($value);
        }
    } else {
        // Fallback: look for formData[key] pattern in $_POST
        foreach ($_POST as $key => $value) {
            if (strpos($key, 'formData[') === 0) {
                $field_name = str_replace(array('formData[', ']'), '', $key);
                $form_data_raw[sanitize_key($field_name)] = sanitize_textarea_field($value);
            }
        }
    }
    
    // Create post title with form ID and timestamp - sanitize for safety
    $post_title = sprintf('Form Submission: %s - %s', 
        sanitize_text_field($form_id), 
        sanitize_text_field(date('Y-m-d H:i:s', $timestamp / 1000))
    );
    
    // Sanitize all meta values before storing
    $sanitized_meta = array(
        '_wvc_form_id' => sanitize_text_field($form_id),
        '_wvc_section_name' => sanitize_text_field($section_name),
        '_wvc_timestamp' => floatval($timestamp), // Ensure numeric value
        '_wvc_session_id' => sanitize_text_field($session_id),
        '_wvc_page_url' => esc_url_raw($page_url),
        '_wvc_user_agent' => sanitize_textarea_field($user_agent), // Can be long, use textarea sanitization
        '_wvc_submission_attempt' => absint($submission_attempt), // Ensure positive integer
        '_wvc_form_version' => sanitize_text_field($form_version),
        '_wvc_validation_errors' => $validation_errors ? sanitize_textarea_field($validation_errors) : '',
        '_wvc_form_data' => wp_json_encode($form_data_raw, JSON_UNESCAPED_UNICODE) // Use wp_json_encode for better security
    );
    
    // Create the post
    $post_data = array(
        'post_title' => sanitize_text_field($post_title),
        'post_content' => '', // We'll store data in meta
        'post_status' => 'private',
        'post_type' => 'wvc_form_data',
        'post_author' => absint(get_current_user_id() ?: 1), // Use current user or fallback to admin
        'meta_input' => $sanitized_meta
    );
    
    // Insert the post
    $post_id = wp_insert_post($post_data, true);
    
    if (is_wp_error($post_id)) {
        wp_send_json_error(array(
            'message' => 'Failed to save form submission: ' . $post_id->get_error_message()
        ), 500);
        return;
    }
    
    // Store individual form fields as separate meta for easier querying
    foreach ($form_data_raw as $field_name => $field_value) {
        $sanitized_field_name = sanitize_key($field_name);
        $sanitized_field_value = sanitize_textarea_field($field_value);
        update_post_meta($post_id, '_wvc_field_' . $sanitized_field_name, $sanitized_field_value);
    }
    
    // Also store the complete form data structure in postmeta for backup/reference
    // Re-sanitize the complete form data array before storing
    $sanitized_form_data_complete = array();
    foreach ($form_data_raw as $key => $value) {
        $sanitized_form_data_complete[sanitize_key($key)] = sanitize_textarea_field($value);
    }
    update_post_meta($post_id, '_wvc_form_data_complete', $sanitized_form_data_complete);

    // Store form fields schema snapshot for this submission
    // This preserves the form structure at the time of submission
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));

    if (!empty($forms)) {
        $form = $forms[0];
        $form_fields_schema = get_post_meta($form->ID, 'wvc_form_fields', true);
        $form_version = get_post_meta($form->ID, 'wvc_form_version', true);

        if (!empty($form_fields_schema)) {
            update_post_meta($post_id, '_wvc_form_fields', $form_fields_schema);

            // Store the form version that was used for this submission
            if (!empty($form_version)) {
                update_post_meta($post_id, '_wvc_form_schema_version', intval($form_version));
            }
        }
    }

    // Store additional metadata as individual post meta entries with sanitization
    $submission_metadata = array(
        'user_ip' => sanitize_text_field($_SERVER['REMOTE_ADDR'] ?? 'unknown'),
        'submission_source' => sanitize_text_field('website_form'),
        'browser_info' => sanitize_textarea_field($user_agent),
        'referrer' => esc_url_raw($_SERVER['HTTP_REFERER'] ?? ''),
        'form_fields_count' => absint(count($form_data_raw))
    );
    update_post_meta($post_id, '_wvc_submission_metadata', $submission_metadata);

    // Send email notification if enabled (non-blocking - don't fail submission if email fails)
    $email_settings = wvc_get_email_notification_settings($form_id);

    if (!empty($email_settings['enabled']) && !empty($email_settings['emails'])) {
        // Attempt to send email notification
        $email_sent = wvc_send_submission_notification_email(
            $post_id,
            $form_id,
            $email_settings['emails']
        );

        // Log if email failed, but don't fail the submission
        if (!$email_sent) {
            error_log(sprintf(
                'WVC Form Submission: Email notification failed for submission #%d (form: %s)',
                $post_id,
                $form_id
            ));
        }
    }

    // Trigger hooks if enabled (non-blocking - don't fail submission if hooks fail)
    $hook_settings = wvc_get_hook_settings($form_id);

    if (!empty($hook_settings['enabled']) && !empty($hook_settings['hooks'])) {
        foreach ($hook_settings['hooks'] as $hook) {
            // Skip disabled hooks
            if (empty($hook['enabled'])) {
                continue;
            }

            // Currently only Slack supported
            if ($hook['type'] === 'slack') {
                $hook_sent = wvc_send_slack_notification(
                    $post_id,
                    $form_id,
                    $hook
                );

                // Log if hook failed
                if (!$hook_sent) {
                    error_log(sprintf(
                        'WVC Form Submission: Slack hook failed for submission #%d (hook: %s)',
                        $post_id,
                        $hook['name'] ?? 'Unnamed'
                    ));
                }
            }
        }
    }

    // Return success response using wp_send_json_success with sanitized values
    wp_send_json_success(array(
        'message' => sanitize_text_field('Form submission saved successfully'),
        'submission_id' => absint($post_id),
        'timestamp' => floatval($timestamp),
        'form_id' => sanitize_text_field($form_id)
    ));
}

/**
 * Validate form field value against validation rules
 *
 * @param mixed $value Field value
 * @param array $field Field configuration with validation rules
 * @return string|null Error message if validation fails, null if passes
 */
function wvc_validate_field_value($value, $field) {
    $validation = isset($field['validation']) ? $field['validation'] : array();

    // Required check
    if (!empty($field['required']) && empty($value)) {
        return sprintf(__('%s is required', 'wvc-theme'), $field['label']);
    }

    // If value is empty and not required, skip other validation
    if (empty($value)) {
        return null;
    }

    // Return early if no validation rules
    if (empty($validation) || !is_array($validation)) {
        return null;
    }

    // Length validation
    if (isset($validation['min_length']) && strlen($value) < $validation['min_length']) {
        return sprintf(
            __('%s must be at least %d characters', 'wvc-theme'),
            $field['label'],
            $validation['min_length']
        );
    }

    if (isset($validation['max_length']) && strlen($value) > $validation['max_length']) {
        return sprintf(
            __('%s must be at most %d characters', 'wvc-theme'),
            $field['label'],
            $validation['max_length']
        );
    }

    // Numeric validation
    if ($field['type'] === 'number') {
        if (isset($validation['min']) && floatval($value) < floatval($validation['min'])) {
            return sprintf(
                __('%s must be at least %s', 'wvc-theme'),
                $field['label'],
                $validation['min']
            );
        }

        if (isset($validation['max']) && floatval($value) > floatval($validation['max'])) {
            return sprintf(
                __('%s must be at most %s', 'wvc-theme'),
                $field['label'],
                $validation['max']
            );
        }
    }

    // Pattern validation
    if (isset($validation['pattern']) && !empty($validation['pattern'])) {
        $pattern = $validation['pattern'];

        if ($pattern === 'email') {
            if (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
                return __('Invalid email address', 'wvc-theme');
            }
        } elseif ($pattern === 'url') {
            if (!filter_var($value, FILTER_VALIDATE_URL)) {
                return __('Invalid URL', 'wvc-theme');
            }
        } elseif ($pattern === 'phone') {
            // Basic phone validation
            if (!preg_match('/^[0-9\s\-\+\(\)]+$/', $value)) {
                return __('Invalid phone number', 'wvc-theme');
            }
        } else {
            // Custom regex pattern
            if (!preg_match('/' . $pattern . '/', $value)) {
                return sprintf(__('Invalid format for %s', 'wvc-theme'), $field['label']);
            }
        }
    }

    return null; // Validation passed
}

/**
 * Sanitize field value based on type
 *
 * @param mixed  $value Field value
 * @param string $type  Field type
 * @return mixed Sanitized value
 */
function wvc_sanitize_field_value($value, $type) {
    switch ($type) {
        case 'email':
            return sanitize_email($value);

        case 'url':
            return esc_url_raw($value);

        case 'number':
            return is_numeric($value) ? floatval($value) : 0;

        case 'textarea':
            return sanitize_textarea_field($value);

        case 'checkbox':
            return $value === '1' || $value === 'on' || $value === true;

        case 'text':
        case 'tel':
        default:
            return sanitize_text_field($value);
    }
}

/**
 * Register AJAX actions for form submission
 */
function wvc_register_ajax_actions() {
    // For logged-in users
    add_action('wp_ajax_wvc_form_submit', __NAMESPACE__ . '\wvc_handle_ajax_form_submission');
    // For non-logged-in users (public forms)
    add_action('wp_ajax_nopriv_wvc_form_submit', __NAMESPACE__ . '\wvc_handle_ajax_form_submission');
}