<?php

/**
 * WVC Import Class
 *
 * Handles importing pages and menus from an S3 bucket.
 * Downloads a zip file, extracts it, and processes its contents to create
 * WordPress pages and menu structures.
 *
 * @package WVC_Theme
 * @subpackage Import
 * @since 1.0.0
 */

class WVC_Import
{
    /**
     * @var string s3 signed url
     */
    private $s3_url = "";

    /**
     * @var string builds dir
     */
    private $builds_dir = "";

    /**
     * @var string builds url
     */
    private $builds_url = "";

    /**
     * @var string current build ID
     */
    private $build_id = "";

    /**
     * @var array meta data from builds dir
     */
    private $meta_data = [];

    /**
     * @var array images data from builds dir
     */
    private $images_data = [];


    /**
     * Constructor
     *
     * @param string $s3_url The signed S3 URL to download the zip file
     * @param string $build_id Build identifier
     */
    public function __construct($s3_url, $build_id)
    {
        $this->s3_url     = $s3_url;
        $wp_upload_dir    = wp_upload_dir();
        $this->builds_dir = $wp_upload_dir['basedir'] . '/wvc-builds/';
        $this->builds_url = $wp_upload_dir['baseurl'] . '/wvc-builds/';
        if ($build_id) {
            $this->build_id = $build_id;
        } else {
            $this->build_id = 'build_' . time();
        }

        $this->createBuildsDirectory();
    }

    /**
     * Create builds directory if it doesn't exist
     */
    private function createBuildsDirectory()
    {
        if ( ! file_exists($this->builds_dir)) {
            wp_mkdir_p($this->builds_dir);
        }
    }


    /**
     * Start the import process
     *
     * @return array Results of the import process
     */
    public function import()
    {
        $result = [
            'success' => false,
            'message' => '',
            'pages'   => [],
            'menus'   => []
        ];

        try {
            // Download and extract the zip file
            $download_result = $this->downloadFromS3();
            if ( ! $download_result['success']) {
                $result['message'] = $download_result['message'];

                return $result;
            }

            // Parse the meta.json file
            $this->meta_data = $this->parseJsonFile('meta.json');
            if (empty($this->meta_data)) {
                $result['message'] = 'Failed to parse meta.json or file is empty';

                return $result;
            }

            $this->images_data = $this->parseJsonFile('media_list.json');

            // Remove WooCommerce pages before completing import
            $this->removeWooPages();
            $this->setUpWoo();

            $images_result    = $this->createImages();
            $result['images'] = $images_result;

            // Create pages
            $pages_result    = $this->createPages();
            $result['pages'] = $pages_result;

            $templates_result           = $this->createTemplates();
            $result['templates_result'] = $templates_result;

            $stylesheet_result    = $this->createStyleKit();
            $result['stylesheet'] = $stylesheet_result;


            $this->updateSiteTitle();

            $result['success'] = true;
            $result['message'] = 'Import completed successfully';

            $this->clearSiteCache();

        } catch (Exception $e) {
            $result['message'] = 'Error during import: ' . $e->getMessage();
        }

        return $result;
    }

    /**
     * Remove WooCommerce pages from WordPress
     *
     * Checks for WooCommerce page options and removes the corresponding pages
     */
    private function removeWooPages()
    {
        // Check if this is a shop website - if so, don't remove WooCommerce required pages
        $shop_enabled = isset($this->meta_data['website_type']) && $this->meta_data['website_type'] == "ecommerce" ? true : false;

        update_option("wvc_start_woo_pages_removal", true);

        // Check if WooCommerce pages have already been removed
        $woo_reset_flag = get_option("wvc_woo_reset");
        if ($woo_reset_flag) {
            return; // Pages already removed, no need to remove again
        }
        $woo_page_options = [
            "woocommerce_shop_page_id"      => true,
            "woocommerce_cart_page_id"      => true,
            "woocommerce_checkout_page_id"  => true,
            "woocommerce_myaccount_page_id" => true
        ];
        if ($shop_enabled) {
            $woo_page_options["woocommerce_myaccount_page_id"] = false;
        }
        $pages_removed = false;

        foreach ($woo_page_options as $option_name => $delete_flag) {
            $page_id = get_option($option_name);

            if ($delete_flag && $page_id && is_numeric($page_id)) {
                // Convert to integer to ensure proper handling
                $page_id = intval($page_id);

                // Delete the page
                $deleted = wp_delete_post($page_id, true);

                // Remove the option
                delete_option($option_name);
                if ($deleted) {
                    $pages_removed = true;
                } else {
                    update_option("wvc_failed_remval_woo_pages", true);
                }
            } else {
                update_option("wvc_no_woo_pages", true);
            }
        }

        // Set flag if any pages were removed
        if ($pages_removed) {
            update_option("wvc_woo_reset", true);
        }
    }

    private function setUpWoo()
    {
        $shop_enabled = isset($this->meta_data['website_type']) && $this->meta_data['website_type'] == "ecommerce" ? true : false;
        if ( ! $shop_enabled) {
            return;
        }
        //update_option("woocommerce_enable_ajax_add_to_cart", "yes", 1);

        // direct bank transfer option
        $woocommerce_bacs_settings = get_option("woocommerce_bacs_settings");
        $woocommerce_bacs_settings['enabled'] = "yes";
        update_option("woocommerce_bacs_settings", $woocommerce_bacs_settings, 1);

        // check payments option
        $woocommerce_cheque_settings = get_option("woocommerce_cheque_settings");
        $woocommerce_cheque_settings['enabled'] = "yes";
        update_option("woocommerce_cheque_settings", $woocommerce_cheque_settings, 1);

        // cash on delivery option
        $woocommerce_cod_settings = get_option("woocommerce_cod_settings");
        $woocommerce_cod_settings['enabled'] = "yes";
        update_option("woocommerce_cod_settings", $woocommerce_cod_settings, 1);

    }

    /**
     * Download zip from S3 and extract it to builds directory
     *
     * @return array Result of the download and extraction
     */
    private function downloadFromS3()
    {
        $result = [
            'success'    => false,
            'message'    => '',
            'build_path' => ''
        ];

        // Create a unique directory for this build
        $build_path = $this->getBuildPath();
        $zip_file   = $build_path . 'import.zip';

        // Create directory for this build
        wp_mkdir_p($build_path);

        // Download the file

        $download_result = $this->downloadFile($zip_file);
        if ( ! $download_result['success']) {
            $result['message'] = $download_result['message'];

            return $result;
        }

        // Extract the zip file
        $extract_result = $this->extractZipFile($zip_file, $build_path);
        if ( ! $extract_result['success']) {
            $result['message'] = $extract_result['message'];

            return $result;
        }

        $result['success']    = true;
        $result['message']    = 'File downloaded and extracted successfully';
        $result['build_path'] = $build_path;

        return $result;
    }

    /**
     * Get the build path for the current import
     *
     * @return string Build path
     */
    private function getBuildPath()
    {
        return $this->builds_dir . $this->build_id . '/';
    }

    /**
     * Get the build URL for the current import
     *
     * @return string Build URL
     */
    private function getBuildUrl()
    {
        return $this->builds_url . $this->build_id . '/';
    }

    /**
     * Download file from URL
     *
     * @param string $destination_path Path to save the file
     *
     * @return array Result of the download operation
     */
    private function downloadFile($destination_path)
    {
        $result = [
            'success' => false,
            'message' => ''
        ];

        // Download the file from S3
        $response = wp_remote_get($this->s3_url, [
            'timeout'   => 60,
            'sslverify' => false
        ]);

        if (is_wp_error($response)) {
            $result['message'] = 'Failed to download file: ' . $response->get_error_message();

            return $result;
        }

        $http_code = wp_remote_retrieve_response_code($response);
        if ($http_code !== 200) {
            $result['message'] = 'Failed to download file. HTTP code: ' . $http_code;

            return $result;
        }

        // Save the file
        $file_content = wp_remote_retrieve_body($response);
        if (empty($file_content)) {
            $result['message'] = 'Downloaded file is empty';

            return $result;
        }

        $saved = file_put_contents($destination_path, $file_content);
        if ( ! $saved) {
            $result['message'] = 'Failed to save downloaded file';

            return $result;
        }

        $result['success'] = true;

        return $result;
    }

    /**
     * Extract zip file to destination
     *
     * @param string $zip_file Path to the zip file
     * @param string $destination_path Path to extract to
     *
     * @return array Result of the extraction operation
     */
    private function extractZipFile($zip_file, $destination_path)
    {
        $result = [
            'success' => false,
            'message' => ''
        ];

        $zip = new ZipArchive();
        if ($zip->open($zip_file) === true) {
            $zip->extractTo($destination_path);
            $zip->close();

            // Delete the zip file after extraction
            @unlink($zip_file);

            $result['success'] = true;
        } else {
            $result['message'] = 'Failed to extract zip file';
        }

        return $result;
    }

    /**
     * Parse the meta.json file from the builds directory
     *
     * @return array The parsed meta data
     */
    private function parseJsonFile($file_name)
    {
        $meta_file = $this->getBuildPath() . $file_name;

        if ( ! file_exists($meta_file)) {
            return [];
        }

        $meta_json = file_get_contents($meta_file);
        if (empty($meta_json)) {
            return [];
        }

        $meta_data = json_decode($meta_json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [];
        }

        return $meta_data;
    }

    private function createImages()
    {
        $img_upload_path = __DIR__ . '/images-upload.php';
        if (file_exists($img_upload_path)) {
            require_once $img_upload_path;
        }
        if (class_exists('WVC_Images_Upload')) {
            $image_upload = new WVC_Images_Upload($this->images_data);

            $result =  $image_upload->createImages();
            $this->images_data = $image_upload->getImagesData();
            return $result;
        }

        return [
            'created' => 0,
            'updated' => 0,
            'failed'  => 0,
            'images'  => []
        ];
    }

    /**
     * Create pages from the import data
     *
     * @return array Results of page creation
     */
    private function createPages()
    {
        $result = [
            'updated' => 0,
            'failed'  => 0,
            'pages'   => []
        ];

        $pages_data = $this->meta_data['pages'];
        $index      = 0;
        foreach ($pages_data as $page_data) {
            $page_result = $this->processPage($page_data);

            if ($page_result['status'] === 'updated') {
                $result['updated']++;
                $page_id = $page_result['data']['id'];

                $should_become_home = get_post_meta($page_id, '_wvc_home', true);
                if ($should_become_home) {
                    update_option('show_on_front', 'page');
                    update_option('page_on_front', $page_id);
                    delete_post_meta($page_id, '_wvc_home');
                }

                $should_become_checkout = get_post_meta($page_id, '_wvc_checkout', true);
                if ($should_become_checkout) {
                    update_option('woocommerce_checkout_page_id', $page_id);
                    delete_post_meta($page_id, '_wvc_checkout');
                }

                $should_become_cart = get_post_meta($page_id, '_wvc_cart', true);
                if ($should_become_cart) {
                    update_option('woocommerce_cart_page_id', $page_id);
                    delete_post_meta($page_id, '_wvc_cart');
                }

                $index++;
            } elseif ($page_result['status'] === 'failed') {
                $result['failed']++;
            }
            $result['pages'][] = $page_result['data'];
        }

        if ($result['updated'] > 0) {
            delete_option("wvc_empty");
        }

        return $result;
    }

    /**
     * Process a single page by data
     *
     * @param  $page_data
     *
     * @return array Result of processing the page
     */
    private function processPage($page_data)
    {
        $result = [
            'status' => 'failed',
            'data'   => [
                'page_title'     => $page_data["title"],
                'component_name' => $page_data["component_name"],
                'status'         => 'failed',
                'message'        => '',
                'id'             => $page_data["id"]
            ]
        ];

        // Get page content-managers and assets
        $page_content = $this->getPageContent($page_data["title"], $page_data["component_name"]);
        if ($page_content === false && isset($result['data']['message'])) {
            return $result;
        }

        $existing_page_id = null;
        if (empty($page_data["id"])) {
            // Check if page already exists
            $existing_page = $this->findExistingPageByTitle($page_data["title"]);
            if ($existing_page) {
                $existing_page_id = $existing_page->ID;
            }
        } else {
            $existing_page_id = $page_data["id"];
        }

        // Create or update the page
        $page_id = $this->createOrUpdatePage($page_data["title"], $page_content, $existing_page_id);

        if (is_wp_error($page_id)) {
            $result['data']['message'] = $page_id->get_error_message();

            return $result;
        }

        // Update page metadata
        $this->updatePageMeta($page_id, $page_data["component_name"]);

        $result['status']         = $existing_page_id ? 'updated' : 'created';
        $result['data']['id']     = $page_id;
        $result['data']['status'] = $existing_page_id ? 'updated' : 'created';

        return $result;
    }

    /**
     * Get page content-managers from HTML file or create default content-managers
     *
     * @param string $page_title The page title
     * @param string $component_name The component title
     *
     * @return string|false Page content-managers or false on failure
     */
    private function getPageContent($page_title, $component_name)
    {
        $last_page_pos = strrpos($component_name, "Page");
        if ($last_page_pos !== false && $last_page_pos + 4 === strlen($component_name)) {
            $component_name = substr($component_name, 0, $last_page_pos);
        }
        $build_path            = $this->getBuildPath();
        $component_name_pascal = $this->toPascalCase($component_name);
        $page_path             = $build_path . $component_name_pascal . '/';
        $html_file             = $page_path . $component_name_pascal . '.html';
        $id                    = strtolower(str_replace(" ", "-", $page_title));

        $default_html = "<div id='" . $id . "-root'></div>
                    <script>
                        window." . $component_name . "VC.mount(document.getElementById('" . $id . "-root'))
                    </script>";
        if ( ! file_exists($html_file)) {
            // Use default content-managers
            return $default_html;
        }
        // Get page content-managers
        $page_content = file_get_contents($html_file);
        $page_content = str_replace("[woocommerce_cart]", "<!-- [woocommerce_cart] -->", $page_content);
        if (empty($page_content)) {
            return $default_html;
        }

        return $page_content;
    }

    /**
     * Find existing page by import ID
     *
     * @param string page_component The page component (used as import ID)
     *
     * @return WP_Post|null The existing page or null
     */
    private function findExistingPageByTitle($page_component)
    {
        $existing_args = [
            'post_type'      => 'page',
            'posts_per_page' => 1,
            'meta_query'     => [
                [
                    'key'   => '_wvc_imported_id',
                    'value' => sanitize_text_field($page_component)
                ]
            ]
        ];

        $existing_query = new WP_Query($existing_args);
        if ($existing_query->have_posts()) {
            return $existing_query->posts[0];
        }

        return null;
    }

    /**
     * Create or update a page
     *
     * @param string $page_title The page title
     * @param string $page_content The page content-managers
     * @param WP_Post|null $existing_page_id The existing page if any
     *
     * @return int|WP_Error The page ID or WP_Error on failure
     */
    private function createOrUpdatePage($page_title, $page_content, $existing_page_id = null)
    {
        $page_data = [
            'post_title'   => $page_title,
            'post_content' => $this->replaceUrls($page_content),
            'post_status'  => 'publish',
            'post_type'    => 'page',
            'post_parent'  => 0
        ];

        if ($existing_page_id) {
            $page_data['ID'] = $existing_page_id;

            return wp_update_post($page_data);
        } else {
            return wp_insert_post($page_data);
        }
    }

    /**
     * Update page meta data with assets and import info
     *
     * @param int $page_id The page ID
     * @param string $page_component The page component
     */
    private function updatePageMeta($page_id, $page_component_name)
    {

        $last_page_pos = strrpos($page_component_name, "Page");
        if ($last_page_pos !== false && $last_page_pos + 4 === strlen($page_component_name)) {
            $page_component_name = substr($page_component_name, 0, $last_page_pos);
        }
        $page_component_pascal = $this->toPascalCase($page_component_name);
        $build_path            = $this->getBuildPath();
        $build_url             = $this->getBuildUrl();

        $page_path = $build_path . $page_component_pascal . '/';
        $page_url  = $build_url . $page_component_pascal . '/';

        // Store imported ID in post meta
        update_post_meta($page_id, '_wvc_imported_id', sanitize_text_field($page_component_name));
        update_post_meta($page_id, '_wvc_import_build', $this->build_id);

        \WVC\Core\Core::set_wvc_editor_meta($page_id, true);
        // Store JS and CSS in post meta if they exist
        $this->storeAssetFile($page_id, $page_path, $page_url, $page_component_name, 'js');
        $this->storeAssetFile($page_id, $page_path, $page_url, $page_component_name, 'css');
    }


    private function createTemplates()
    {
        $result = [
            'created'   => 0,
            'updated'   => 0,
            'failed'    => 0,
            'templates' => []
        ];
        if ( ! isset($this->meta_data['templates']) && ! isset($this->meta_data['pageTemplates'])) {
            return $result;
        }
        $page_templates     = $this->meta_data['pageTemplates'] ?? [];
        $reusable_templates = $this->meta_data['templates'] ?? [];

        $templates_data = array_merge($page_templates, $reusable_templates);

        foreach ($templates_data as $template_key => $template_data) {
            $template_result = $this->processTemplateByData($template_data, $template_key);

            if ($template_result['status'] === 'created') {
                $result['created']++;
            } elseif ($template_result['status'] === 'updated') {
                $result['updated']++;
            } elseif ($template_result['status'] === 'failed') {
                $result['failed']++;
            }

            $result['templates'][] = $template_result['data'];
        }

        return $result;
    }

    /**
     * Process a single template by data
     *
     * @param array $template_data The template data
     * @param string $template_key The template key from meta.json
     *
     * @return array Result of processing the template
     */
    private function processTemplateByData($template_data, $template_key)
    {
        $result = [
            'status' => 'failed',
            'data'   => [
                'template_title' => $template_data["title"] ?? $template_key,
                'template_key'   => $template_key,
                'status'         => 'failed',
                'message'        => '',
                'id'             => 0
            ]
        ];

        // Get template content and assets
        $template_content = $this->getTemplateContent($template_key);
        if ($template_content === false) {
            $result['data']['message'] = 'Failed to get template content';

            return $result;
        }

        // Check if template already exists by title
        $existing_template    = $this->findExistingTemplateByTitle($template_data["title"] ?? $template_key);
        $existing_template_id = $existing_template ? $existing_template->ID : null;

        // Create or update the template
        $template_id = $this->createOrUpdateTemplate(
            $template_data["title"] ?? $template_key,
            $template_content,
            $existing_template_id
        );

        if (is_wp_error($template_id)) {
            $result['data']['message'] = $template_id->get_error_message();

            return $result;
        }

        // Update template metadata
        $this->updateTemplateMeta($template_id, $template_data["title"] ?? $template_key, $template_key);

        $result['status']         = $existing_template_id ? 'updated' : 'created';
        $result['data']['id']     = $template_id;
        $result['data']['status'] = $existing_template_id ? 'updated' : 'created';

        return $result;
    }

    /**
     * Get template content from HTML file or create default content
     *
     * @param string $template_key The template key
     *
     * @return string|false Template content or false on failure
     */
    private function getTemplateContent($template_key)
    {
        $build_path                    = $this->getBuildPath();
        $template_key_no_spaces        = str_replace(' ', '', $template_key);
        $template_key_no_spaces_pascal = $this->toPascalCase($template_key_no_spaces);
        $template_path                 = $build_path . $template_key_no_spaces_pascal . '/';
        $html_file                     = $template_path . $template_key_no_spaces_pascal . '.html';

        if ( ! file_exists($html_file)) {
            // Use default content
            return false;
        }

        // Get template content
        $template_content = file_get_contents($html_file);
        if (empty($template_content)) {
            return '<div id="root_' . sanitize_title($template_key) . '" class="wvc-template-' . sanitize_title($template_key) . '">' .
                   '<h2>' . esc_html(ucfirst($template_key)) . ' Template</h2>' .
                   '<p>This is the ' . esc_html($template_key) . ' template content.</p>' .
                   '</div>';
        }

        return $template_content;
    }

    /**
     * Find existing template by title
     *
     * @param string $template_title The template title
     *
     * @return WP_Post|null The existing template post or null
     */
    private function findExistingTemplateByTitle($template_title)
    {
        $existing_args = [
            'post_type'      => 'wvc_template',
            'post_status'    => 'any',
            'posts_per_page' => -1,
            'meta_query'     => [
                [
                    'key'   => '_wvc_imported_id',
                    'value' => sanitize_text_field($template_title)
                ]
            ],
            'orderby'        => 'ID',
            'order'          => 'ASC'
        ];

        $existing_query = new WP_Query($existing_args);
        if ($existing_query->have_posts()) {
            // Return the template with the lowest ID if multiple exist
            return $existing_query->posts[0];
        }

        return null;
    }

    /**
     * Create or update a template
     *
     * @param string $template_title The template title
     * @param string $template_content The template content
     * @param int|null $existing_template_id The existing template ID if any
     *
     * @return int|WP_Error The template ID or WP_Error on failure
     */
    private function createOrUpdateTemplate($template_title, $template_content, $existing_template_id = null)
    {
        $template_data = [
            'post_title'   => $template_title,
            'post_content' => $this->replaceUrls($template_content),
            'post_status'  => 'publish',
            'post_type'    => 'wvc_template',
            'post_parent'  => 0
        ];

        if ($existing_template_id) {
            $template_data['ID'] = $existing_template_id;

            return wp_update_post($template_data);
        } else {
            return wp_insert_post($template_data);
        }
    }

    /**
     * Update template meta data with assets and import info
     *
     * @param int $template_id The template ID
     * @param string $template_title The template title
     * @param string $template_key The template key from meta.json
     */
    private function updateTemplateMeta($template_id, $template_title, $template_key)
    {
        $build_path                    = $this->getBuildPath();
        $build_url                     = $this->getBuildUrl();
        $template_key_no_spaces        = str_replace(' ', '', $template_key);
        $template_key_no_spaces_pascal = $this->toPascalCase($template_key_no_spaces);


        $template_path = $build_path . $template_key_no_spaces_pascal . '/';
        $template_url  = $build_url . $template_key_no_spaces . '/';

        // Store imported ID in post meta
        update_post_meta($template_id, '_wvc_imported_id', sanitize_text_field($template_title));
        update_post_meta($template_id, '_wvc_import_build', $this->build_id);
        update_post_meta($template_id, '_wvc_template_key', sanitize_text_field($template_key));

        // Set template type based on template key
        $template_type = $this->getTemplateTypeFromKey($template_key);

        update_post_meta($template_id, '_wvc_template_type', $template_type);

        \WVC\Core\Core::set_wvc_editor_meta($template_id, true);

        // Store JS and CSS in post meta if they exist
        $this->storeAssetFile($template_id, $template_path, $template_url, $template_key, 'js');
        $this->storeAssetFile($template_id, $template_path, $template_url, $template_key, 'css');
    }

    /**
     * Get template type from template key
     *
     * @param string $template_key The template key from meta.json
     *
     * @return string The template type
     */
    private function getTemplateTypeFromKey($template_key)
    {
        $template_key_lower = strtolower($template_key);

        // Handle Single*Template pattern
        // SinglePostTemplate -> single
        // SingleTestimonialTemplate -> single_testimonial
        // SingleTeamMemberTemplate -> single_team_member
        if (strpos($template_key_lower, 'single') === 0 && strpos($template_key_lower, 'template') !== false) {
            if ($template_key_lower === 'singleposttemplate') {
                return 'single';
            } else {
                // Extract CPT name from original key (preserve camelCase)
                $cpt_name = str_replace(['Single', 'Template'], '', $template_key);
                // Convert camelCase/PascalCase to snake_case
                $cpt_name_snake = $this->camelToSnakeCase($cpt_name);

                return "single_{$cpt_name_snake}";
            }
        }

        // Handle Archive*Template pattern
        // Archive -> archive
        // ArchivePost -> archive
        // ArchiveProduct -> archive_product
        // ArchiveTestimonial -> archive_testimonial
        if (strpos($template_key_lower, 'archive') === 0) {
            if ($template_key_lower === 'archive' || $template_key_lower === 'archivepost') {
                return 'archive';
            } else {
                // Extract CPT name from original key (preserve camelCase)
                $cpt_name = str_replace(['Archive', 'Template'], '', $template_key);
                // Convert camelCase/PascalCase to snake_case
                $cpt_name_snake = $this->camelToSnakeCase($cpt_name);

                return "archive_{$cpt_name_snake}";
            }
        }

        // Map template keys to template types
        $type_mapping = array(
            'header'  => 'header',
            'footer'  => 'footer',
            'sidebar' => 'sidebar',
            'content' => 'content',
        );

        return isset($type_mapping[$template_key_lower]) ? $type_mapping[$template_key_lower] : 'content';
    }

    /**
     * Convert camelCase or PascalCase to snake_case
     *
     * @param string $string The string to convert
     *
     * @return string The snake_case string
     */
    private function camelToSnakeCase($string)
    {
        // Insert underscore before uppercase letters (except the first one)
        $snake = preg_replace('/([a-z])([A-Z])/', '$1_$2', $string);
        // Convert to lowercase
        return strtolower($snake);
    }

    /**
     * Store asset file in post meta
     *
     * @param int $page_id The page ID
     * @param string $page_path The page directory path
     * @param string $page_url The page directory URL
     * @param string $page_component The page component
     * @param string $type The asset type (js or css)
     */
    private function storeAssetFile($page_id, $page_path, $page_url, $page_component, $type)
    {
        $page_component_pascal = $this->toPascalCase($page_component);
        $file                  = $page_path . $page_component_pascal . '.' . $type;
        $url                   = $page_url . $page_component . '.' . $type;

        if (file_exists($file)) {
            $content = file_get_contents($file);
            $content = $this->replaceUrls($content, "js");
            # important - we add slashes because wp removes them during update_post_meta under the hood
            # otherwise it breaks the js/css in meta
            update_post_meta($page_id, '_wvc_page_' . $type, wp_slash($content));
            update_post_meta($page_id, '_wvc_page_' . $type . '_url', $url);
        }
    }

    private function replaceUrls($content, $type = null)
    {

        # replace image urls with wp_src
        $images_data = $this->images_data;

        foreach ($images_data as $image_key => $image_data) {
            $content = str_replace($image_data['cloud_src'], $image_data['wp_src'], $content);

            # replace tailwind-escaped urls as well, like those in css classes , e.g.
            #    .bg-\[url\(https\:\/\/wpvc-images\.s3\.us-east-1\.amazonaws\.com\/images\/21921\/img\/hero_background\.png\)\] {

            $content = str_replace(preg_replace('/([^a-zA-Z0-9_-])/', '\\\\$1', $image_data['cloud_src']),
                preg_replace('/([^a-zA-Z0-9_-])/', '\\\\$1', $image_data['wp_src']), $content);
        }

        # search for other urls in the content
        # 1. root url href="/" or href='/', which should be replaced with website url

        $count   = 0;
        // Replace href="/" patterns (with optional hash fragments)
        // Matches: href="/", href="/#", href="/#something"
        $content = preg_replace('/href\s*=\s*["\']\s*\/(#[^"\']*)?["\']/is', 'href="' . home_url() . '$1"', $content, -1, $count);

        if ($type == "js") {
            # handle also patterns like to:"/" (with optional hash fragments)
            # Matches: to:"/", to:"/#", to:"/#something"

            $content = preg_replace(
                '/to\s*:\s*["\']\s*\/(#[^"\']*)?["\']/is',
                'to:"' . home_url() . '$1"',
                $content,
                -1,
                $count
            );
        }

        # 2. replace page urls like href="/BlogPage" with WordPress page URLs using IDs
        if (isset($this->meta_data['pages']) && ! empty($this->meta_data['pages'])) {
            // Pre-cache all page URLs to avoid multiple get_permalink() calls
            $page_urls          = [];
            $replacement_counts = [];

            foreach ($this->meta_data['pages'] as $page_data) {
                $page_id = $page_data['id'];
                if ( ! empty($page_id)) {
                    $page_urls[$page_data['component_name']]          = get_permalink($page_id);
                    $replacement_counts[$page_data['component_name']] = 0;
                }
            }

            // Now replace URLs using cached values
            foreach ($page_urls as $component_name => $wp_page_url) {
                $count = 0;

                // Replace href="/ComponentName" patterns (with optional trailing slash and hash fragments)
                // Matches: /ComponentName, /ComponentName/, /ComponentName#, /ComponentName/#, /ComponentName/#something
                $content                             = preg_replace(
                    '/href\s*=\s*["\']\s*\/' . preg_quote($component_name, '/') . '(?:\/)?(#[^"\']*)?["\']/is',
                    'href="' . $wp_page_url . '$1"',
                    $content,
                    -1,
                    $count
                );
                $replacement_counts[$component_name] += $count;

                if ($type == "js") {

                    # handle also patterns like to:"/BlogPage" (with optional trailing slash and hash fragments)
                    # Matches: to:"/ComponentName", to:"/ComponentName/", to:"/ComponentName#", to:"/ComponentName/#", to:"/ComponentName/#something"
                    $content = preg_replace(
                        '/to:\s*["\']\s*\/' . preg_quote($component_name, '/') . '(?:\/)?(#[^"\']*)?["\']/is',
                        'to:"' . $wp_page_url . '$1"',
                        $content,
                        -1,
                        $count
                    );

                    # handle also patterns like href:"/BlogPage" (with optional trailing slash and hash fragments)
                    # Matches: href:"/ComponentName", href:"/ComponentName/", href:"/ComponentName#", href:"/ComponentName/#", href:"/ComponentName/#something"
                    $content = preg_replace(
                        '/href:\s*["\']\s*\/' . preg_quote($component_name, '/') . '(?:\/)?(#[^"\']*)?["\']/is',
                        'href:"' . $wp_page_url . '$1"',
                        $content,
                        -1,
                        $count
                    );

                    # handle also patterns like link:"/BlogPage" (with optional trailing slash and hash fragments)
                    # this occasionally happens if there is a listicle template-based component with URL under link property of an object
                    # Matches: link:"/ComponentName", link:"/ComponentName/", link:"/ComponentName#", link:"/ComponentName/#", link:"/ComponentName/#something"
                    $content = preg_replace(
                        '/link:\s*["\']\s*\/' . preg_quote($component_name, '/') . '(?:\/)?(#[^"\']*)?["\']/is',
                        'link:"' . $wp_page_url . '$1"',
                        $content,
                        -1,
                        $count
                    );

                }
            }
        }

        return $content;
    }

    private function createStyleKit()
    {
        $result = [
            "created" => 0,
            "updated" => 0,
            "failed"  => 0
        ];

        // stylesheet file index.css 
        $stylesheet_file = $this->getBuildPath() . "index.css";
        if (file_exists($stylesheet_file)) {
            $stylesheet_content = file_get_contents($stylesheet_file);

            if ( ! empty($stylesheet_content)) {
                // Check if a style kit already exists
                $existing_kit = $this->getExistingStyleKit();

                if ($existing_kit) {
                    // Update existing style kit
                    $updated = $this->updateStyleKit($existing_kit->ID, $stylesheet_content);
                    if ($updated) {
                        $result["updated"]++;
                        $result["message"] = "Style kit updated successfully";
                    } else {
                        $result["failed"]++;
                        $result["message"] = "Failed to update style kit";
                    }
                } else {
                    // Create new style kit
                    $created = $this->createNewStyleKit($stylesheet_content);
                    if ($created) {
                        $result["created"]++;
                        $result["message"] = "Style kit created successfully";
                    } else {
                        $result["failed"]++;
                        $result["message"] = "Failed to create style kit";
                    }
                }
            } else {
                $result["failed"]++;
                $result["message"] = "Stylesheet file is empty";
            }
        } else {
            $result["failed"]++;
            $result["message"] = "Stylesheet file not found";
        }

        return $result;
    }

    /**
     * Get existing style kit if any
     */
    private function getExistingStyleKit()
    {
        $args = [
            "post_type"      => "style_kit",
            "post_status"    => "publish",
            "posts_per_page" => 1,
            "orderby"        => "date",
            "order"          => "DESC"
        ];

        $query = new WP_Query($args);

        if ($query->have_posts()) {
            return $query->posts[0];
        }

        return null;
    }

    /**
     * Create new style kit
     */
    private function createNewStyleKit($css_content)
    {
        // Prepare post data
        $post_data = [
            "post_title"   => "Imported Style Kit - " . date("Y-m-d H:i:s"),
            "post_type"    => "style_kit",
            "post_status"  => "publish", // Start as alternative, can be published later
            "post_content" => "Auto-generated style kit from imported stylesheet"
        ];

        // Insert the post
        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return false;
        }

        // Save CSS content as post meta
        update_post_meta($post_id, "_wvc_css_content", $css_content);

        // Save version
        update_post_meta($post_id, "_wvc_version", "1.0.0");

        return $post_id;
    }

    /**
     * Update existing style kit
     */
    private function updateStyleKit($post_id, $css_content)
    {
        // Update CSS content
        $updated = update_post_meta($post_id, "_wvc_css_content", $css_content);

        if ($updated) {
            // Update the post title to indicate it was updated
            $post_data = [
                "ID"           => $post_id,
                "post_title"   => "Updated Style Kit - " . date("Y-m-d H:i:s"),
                "post_content" => "Auto-updated style kit from imported stylesheet"
            ];
            wp_update_post($post_data);
        }

        return $updated;
    }

    private function updateSiteTitle()
    {

        $site_title = isset($this->meta_data['site_title']) ? $this->meta_data['site_title'] : '';

        if ( ! empty($site_title)) {
            update_option('blogname', $site_title);
        }

        if (
            class_exists('\Tenweb_Authorization\Helper')
            && method_exists('\Tenweb_Authorization\Helper', "check_site_state")
        ) {
            \Tenweb_Authorization\Helper::check_site_state(true);
        }
    }

    private function clearSiteCache($flush_rewrite = true, $regenerate_home_critical = true)
    {
        if ($flush_rewrite) {
            flush_rewrite_rules();//phpcs:ignore WordPressVIPMinimum.Functions.RestrictedFunctions.flush_rewrite_rules_flush_rewrite_rules
        }

        if (class_exists('\TenWebOptimizer\OptimizerAdmin')) {
            \TenWebOptimizer\OptimizerAdmin::get_instance();
            global $TwoSettings;
            $two_critical_pages = $TwoSettings->get_settings("two_critical_pages");

            if ($regenerate_home_critical && ! empty($two_critical_pages["front_page"])) {
                $two_critical_pages["front_page"]["wait_until"] = "load";
                $TwoSettings->update_setting("two_critical_pages", $two_critical_pages);
                \TenWebOptimizer\OptimizerUtils::regenerate_critical("front_page");

            } else {
                \TenWebOptimizer\OptimizerAdmin::clear_cache(
                    false,
                    true,
                    true,
                    true,
                    'front_page',
                    false,
                    true,
                    false,
                    false);
            }
        } else {
            update_option("wvc_purge_cache_called", true);
            update_option("wvc_purge_cache_has_registered_actions", has_action("tenweb_purge_all_caches"));
            do_action('tenweb_purge_all_caches');
        }
    }

    /**
     * Convert a string to PascalCase
     *
     * @param string $string The string to convert
     *
     * @return string The PascalCase string
     */
    private function toPascalCase($string)
    {
        // Remove special characters and split by spaces, hyphens, underscores
        $words = preg_split('/[\s\-_]+/', $string);

        // Capitalize first letter of each word and join
        $pascalCase = '';
        foreach ($words as $word) {
            if ( ! empty($word)) {
                $pascalCase .= ucfirst($word);
            }
        }

        return $pascalCase;
    }
}
