<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * WVC SVG Sanitizer
 *
 * Basic SVG sanitizer for cleaning SVG files before upload.
 * This is a fallback sanitizer that removes potentially dangerous elements.
 *
 * @package WVC_Theme
 * @subpackage Import
 * @since 1.0.0
 */
class WVC_SVG_Sanitizer {

    /**
     * Allowed SVG elements
     *
     * @var array
     */
    private $allowed_elements = [
        'svg', 'path', 'circle', 'ellipse', 'line', 'polyline', 'polygon',
        'rect', 'g', 'defs', 'clipPath', 'mask', 'pattern', 'symbol',
        'text', 'tspan', 'title', 'desc', 'metadata', 'style'
    ];

    /**
     * Allowed SVG attributes
     *
     * @var array
     */
    private $allowed_attributes = [
        'id', 'class', 'style', 'width', 'height', 'viewBox', 'fill',
        'stroke', 'stroke-width', 'stroke-linecap', 'stroke-linejoin',
        'stroke-dasharray', 'stroke-dashoffset', 'opacity', 'transform',
        'x', 'y', 'cx', 'cy', 'r', 'rx', 'ry', 'd', 'points', 'x1', 'y1',
        'x2', 'y2', 'font-family', 'font-size', 'font-weight', 'text-anchor',
        'dominant-baseline', 'text-decoration', 'letter-spacing', 'word-spacing'
    ];

    /**
     * Sanitize SVG file content
     *
     * @param string $content SVG file content
     * @return string|null Sanitized content or null on failure
     */
    public function sanitize_file($content) {
        if (empty($content)) {
            return null;
        }

        // Basic validation - check if it's actually an SVG
        if (strpos($content, '<svg') === false) {
            return null;
        }

        // Remove any PHP tags or script tags
        $content = preg_replace('/<\?php.*?\?>/is', '', $content);
        $content = preg_replace('/<script.*?<\/script>/is', '', $content);

        // Remove any on* event handlers
        $content = preg_replace('/\s+on\w+\s*=\s*["\'][^"\']*["\']/i', '', $content);

        // Remove any javascript: URLs
        $content = preg_replace('/javascript:/i', '', $content);

        // Remove any data: URLs that might contain scripts
        $content = preg_replace('/data:text\/html/i', '', $content);

        // Basic element filtering (this is a simplified approach)
        // In a production environment, you might want to use a more robust XML parser
        
        // Remove any elements that are not in our allowed list
        $dom = new DOMDocument();
        libxml_use_internal_errors(true);
        
        if (!$dom->loadXML($content, LIBXML_NOENT | LIBXML_NOCDATA)) {
            // If XML parsing fails, try with HTML mode
            if (!$dom->loadHTML($content, LIBXML_NOENT | LIBXML_NOCDATA)) {
                return null;
            }
        }
        
        libxml_clear_errors();

        // Get the root element
        $root = $dom->documentElement;
        if (!$root || $root->nodeName !== 'svg') {
            return null;
        }

        // Clean the SVG
        $this->cleanNode($root);

        // Return the sanitized content
        return $dom->saveXML($root);
    }

    /**
     * Clean a DOM node recursively
     *
     * @param DOMNode $node The node to clean
     */
    private function cleanNode($node) {
        if ($node->nodeType === XML_ELEMENT_NODE) {
            $nodeName = strtolower($node->nodeName);
            
            // Remove disallowed elements
            if (!in_array($nodeName, $this->allowed_elements)) {
                $node->parentNode->removeChild($node);
                return;
            }

            // Clean attributes
            $attributes = $node->attributes;
            if ($attributes) {
                $toRemove = [];
                for ($i = 0; $i < $attributes->length; $i++) {
                    $attr = $attributes->item($i);
                    $attrName = strtolower($attr->nodeName);
                    
                    // Remove disallowed attributes
                    if (!in_array($attrName, $this->allowed_attributes)) {
                        $toRemove[] = $attrName;
                    }
                    
                    // Remove any attribute that contains javascript
                    if (stripos($attr->nodeValue, 'javascript:') !== false) {
                        $toRemove[] = $attrName;
                    }
                }
                
                foreach ($toRemove as $attrName) {
                    $node->removeAttribute($attrName);
                }
            }
        }

        // Process child nodes
        $children = $node->childNodes;
        for ($i = $children->length - 1; $i >= 0; $i--) {
            $child = $children->item($i);
            $this->cleanNode($child);
        }
    }
}
