<?php
/**
 * Custom Post Type REST API Class
 *
 * Provides REST API endpoints for registering and managing custom post types.
 * Simplified version with fixed defaults and universal JSON meta key.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    2.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';

/**
 * Class WVC_Custom_Post_Type_REST_API
 *
 * Handles custom post type registration and management via REST API
 */
class WVC_Custom_Post_Type_REST_API extends WVC_REST_API
{
    const MANAGE_OPTIONS_CAP = "manage_options";

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        // Register meta for existing CPTs on init (ensures meta is available after reloads)
        add_action('init', array($this, 'init_register_existing_metas'), 99);
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Register custom post type endpoint
        register_rest_route(
            $this->namespace,
            '/custom-post-types',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'register_custom_post_type'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::MANAGE_OPTIONS_CAP);
                },
                'args'                => array(
                    'post_type_key'    => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_key',
                        'description'       => __('Unique key for the post type (max 20 chars)', 'wvc-theme'),
                        'validate_callback' => function($param, $request, $key) {
                            return strlen($param) <= 20 && preg_match('/^[a-z_]+$/', $param);
                        },
                    ),
                    'post_type_label'  => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('Plural label for the post type', 'wvc-theme'),
                    ),
                    'singular_label'   => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('Singular label for the post type', 'wvc-theme'),
                    ),
                    'fields'           => array(
                        'required'          => false,
                        'type'              => 'array',
                        'default'           => array(),
                        'description'       => __('Custom field schema definitions for dynamic meta box rendering', 'wvc-theme'),
                        'items'             => array(
                            'type'       => 'object',
                            'properties' => array(
                                'key'           => array('type' => 'string'),
                                'label'         => array('type' => 'string'),
                                'type'          => array('type' => 'string'),
                                'required'      => array('type' => 'boolean'),
                                'placeholder'   => array('type' => 'string'),
                                'help_text'     => array('type' => 'string'),
                                'default_value' => array('type' => 'mixed'),
                                'options'       => array('type' => 'array'),
                                'validation'    => array(
                                    'type'       => 'object',
                                    'properties' => array(
                                        'min_length' => array('type' => 'integer'),
                                        'max_length' => array('type' => 'integer'),
                                        'min'        => array('type' => 'number'),
                                        'max'        => array('type' => 'number'),
                                        'pattern'    => array('type' => 'string'),
                                    ),
                                ),
                            ),
                        ),
                    ),
                ),
            )
        );

        // Get all custom post types endpoint
        register_rest_route(
            $this->namespace,
            '/custom-post-types',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_custom_post_types'),
                'permission_callback' => '__return_true', // Public access
            )
        );

        // Unregister custom post type endpoint
        register_rest_route(
            $this->namespace,
            '/custom-post-types/(?P<post_type_key>[a-z_]+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array($this, 'unregister_custom_post_type'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::MANAGE_OPTIONS_CAP);
                },
                'args'                => array(
                    'post_type_key' => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_key',
                        'description'       => __('The post type key to unregister', 'wvc-theme'),
                    ),
                ),
            )
        );
    }

    /**
     * Register a new custom post type
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function register_custom_post_type($request)
    {
        $post_type_key   = $request->get_param('post_type_key');
        $post_type_label = $request->get_param('post_type_label');
        $singular_label  = $request->get_param('singular_label');
        $fields          = $request->get_param('fields') ?: array();

        // Check if post type already exists
        if (post_type_exists($post_type_key)) {
            return new WP_Error(
                'cpt_exists',
                __('Custom post type "' . $post_type_key . '" already exists', 'wvc-theme'),
                array('status' => 400)
            );
        }

        // Sanitize and validate fields array
        $sanitized_fields = $this->sanitize_fields($fields);

        // Save CPT configuration to option for persistence
        $cpt_configs = get_option('wvc_custom_post_types', array());
        $cpt_configs[$post_type_key] = array(
            'post_type_label' => $post_type_label,
            'singular_label'  => $singular_label,
            'fields'          => $sanitized_fields,
        );
        update_option('wvc_custom_post_types', $cpt_configs);

        // Store field schemas separately for easy access by meta box
        if (!empty($sanitized_fields)) {
            update_option("wvc_cpt_fields_{$post_type_key}", $sanitized_fields);
        }

        // Register the post type with fixed defaults
        $args = array(
            'labels'              => array(
                'name'               => $post_type_label,
                'singular_name'      => $singular_label,
                'add_new'            => sprintf(__('Add New %s', 'wvc-theme'), $singular_label),
                'add_new_item'       => sprintf(__('Add New %s', 'wvc-theme'), $singular_label),
                'edit_item'          => sprintf(__('Edit %s', 'wvc-theme'), $singular_label),
                'new_item'           => sprintf(__('New %s', 'wvc-theme'), $singular_label),
                'view_item'          => sprintf(__('View %s', 'wvc-theme'), $singular_label),
                'search_items'       => sprintf(__('Search %s', 'wvc-theme'), $post_type_label),
                'not_found'          => sprintf(__('No %s found', 'wvc-theme'), strtolower($post_type_label)),
                'not_found_in_trash' => sprintf(__('No %s found in Trash', 'wvc-theme'), strtolower($post_type_label)),
                'all_items'          => sprintf(__('All %s', 'wvc-theme'), $post_type_label),
            ),
            'public'              => true,
            'publicly_queryable'  => true,
            'show_ui'             => true,
            'show_in_menu'        => true,  // ✅ Visible in admin menu
            'query_var'           => true,
            'rewrite'             => array('slug' => $post_type_key),
            'capability_type'     => 'post',
            'has_archive'         => true,  // ✅ Fixed default
            'hierarchical'        => false, // ✅ Fixed default
            'menu_position'       => 20,    // After Pages
            'supports'            => array('title', 'editor', 'thumbnail', 'custom-fields'), // ✅ Fixed defaults + custom-fields for meta
            'show_in_rest'        => true,
        );

        $register_result = register_post_type($post_type_key, $args);

        if (is_wp_error($register_result)) {
            return $register_result;
        }

        // Register individual meta keys for each custom field
        foreach ($sanitized_fields as $field) {
            $meta_key = "wvc_{$post_type_key}_{$field['key']}";
            $meta_type = $this->get_meta_type_for_field($field['type']);

            register_post_meta($post_type_key, $meta_key, array(
                'type'         => $meta_type,
                'description'  => $field['label'],
                'single'       => true,
                'show_in_rest' => true,
            ));
        }

        // Flush rewrite rules
        flush_rewrite_rules();

        return rest_ensure_response(array(
            'success' => true,
            'wp_id'   => 0, // WordPress doesn't provide ID for post types
            'message' => sprintf(__('Custom post type "%s" registered successfully with %d custom fields', 'wvc-theme'), $post_type_label, count($sanitized_fields)),
            'data'    => array(
                'post_type_key'   => $post_type_key,
                'post_type_label' => $post_type_label,
                'singular_label'  => $singular_label,
                'fields_count'    => count($sanitized_fields),
            ),
        ));
    }

    /**
     * Get all registered custom post types
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response
     */
    public function get_custom_post_types($request)
    {
        $cpt_configs = get_option('wvc_custom_post_types', array());

        return rest_ensure_response(array(
            'success' => true,
            'data'    => $cpt_configs,
        ));
    }

    /**
     * Unregister a custom post type
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function unregister_custom_post_type($request)
    {
        $post_type_key = $request->get_param('post_type_key');

        // Check if post type exists
        if (!post_type_exists($post_type_key)) {
            return new WP_Error(
                'cpt_not_exists',
                __('Custom post type does not exist', 'wvc-theme'),
                array('status' => 404)
            );
        }

        // Remove from options
        $cpt_configs = get_option('wvc_custom_post_types', array());
        if (isset($cpt_configs[$post_type_key])) {
            unset($cpt_configs[$post_type_key]);
            update_option('wvc_custom_post_types', $cpt_configs);
        }

        // Unregister the post type
        $result = unregister_post_type($post_type_key);

        if (is_wp_error($result)) {
            return $result;
        }

        // Flush rewrite rules
        flush_rewrite_rules();

        return rest_ensure_response(array(
            'success' => true,
            'message' => sprintf(__('Custom post type "%s" unregistered successfully', 'wvc-theme'), $post_type_key),
        ));
    }

    /**
     * Sanitize field schema array
     *
     * @param array $fields The fields array to sanitize.
     *
     * @return array Sanitized fields array
     */
    private function sanitize_fields($fields)
    {
        if (empty($fields) || !is_array($fields)) {
            return array();
        }

        $sanitized = array();
        $valid_types = array('text', 'textarea', 'url', 'email', 'date', 'number', 'select', 'checkbox', 'image', "another_custom_post_type");

        foreach ($fields as $field) {
            if (!is_array($field) || empty($field['key']) || empty($field['label']) || empty($field['type'])) {
                continue; // Skip invalid field definitions
            }

            // Validate field type
            if (!in_array($field['type'], $valid_types, true)) {
                continue;
            }

            $sanitized_field = array(
                'key'           => sanitize_key($field['key']),
                'label'         => sanitize_text_field($field['label']),
                'type'          => sanitize_text_field($field['type']),
                'required'      => !empty($field['required']),
                'placeholder'   => isset($field['placeholder']) ? sanitize_text_field($field['placeholder']) : '',
                'help_text'     => isset($field['help_text']) ? sanitize_text_field($field['help_text']) : '',
                'default_value' => isset($field['default_value']) ? $field['default_value'] : null,
            );

            // Handle options for select fields
            if ($field['type'] === 'select' && !empty($field['options']) && is_array($field['options'])) {
                $sanitized_field['options'] = array_map('sanitize_text_field', $field['options']);
            }

            $sanitized[] = $sanitized_field;
        }

        return $sanitized;
    }

    /**
     * Re-register meta for existing custom post types on init
     * Ensures meta is available even after theme reload
     */
    public function init_register_existing_metas()
    {
        $cpt_configs = get_option('wvc_custom_post_types', array());

        foreach ($cpt_configs as $post_type_key => $config) {
            if (post_type_exists($post_type_key)) {
                $fields = isset($config['fields']) ? $config['fields'] : array();

                // Register individual meta keys for each custom field
                foreach ($fields as $field) {
                    $meta_key = "wvc_{$post_type_key}_{$field['key']}";
                    $meta_type = $this->get_meta_type_for_field($field['type']);

                    register_post_meta($post_type_key, $meta_key, array(
                        'type'         => $meta_type,
                        'description'  => $field['label'],
                        'single'       => true,
                        'show_in_rest' => true,
                    ));
                }
            }
        }
    }

    /**
     * Get WordPress meta type for a field type
     *
     * @param string $field_type The field type.
     *
     * @return string WordPress meta type (string, integer, or boolean)
     */
    private function get_meta_type_for_field($field_type)
    {
        switch ($field_type) {
            case 'number':
            case 'image':
                return 'integer';
            case 'checkbox':
                return 'boolean';
            case 'another_custom_post_type':
                # in this case, we will store the ID of the another custom post type
                # so we need to return the integer type
                return 'integer';
            default:
                return 'string';
        }
    }
}

// Initialize the Custom Post Type REST API
new WVC_Custom_Post_Type_REST_API();
