<?php
/**
 * Form REST API Class
 *
 * Provides REST API endpoints for registering and managing forms.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';

/**
 * Class WVC_Form_REST_API
 *
 * Handles form registration and management via REST API
 */
class WVC_Form_REST_API extends WVC_REST_API
{
    const MANAGE_OPTIONS_CAP = 'manage_options';

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // POST /wvc/v1/forms - Register a new form
        register_rest_route(
            $this->namespace,
            '/forms',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'register_form'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::MANAGE_OPTIONS_CAP);
                },
                'args'                => array(
                    'form_key'   => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_key',
                        'description'       => __('Unique key for the form', 'wvc-theme'),
                    ),
                    'form_label' => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('Display name for the form', 'wvc-theme'),
                    ),
                    'fields'     => array(
                        'required'    => false,
                        'type'        => 'array',
                        'default'     => array(),
                        'description' => __('Form field specifications', 'wvc-theme'),
                    ),
                ),
            )
        );

        // GET /wvc/v1/forms - Get all forms
        register_rest_route(
            $this->namespace,
            '/forms',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_forms'),
                'permission_callback' => '__return_true', // Public endpoint
            )
        );

        // GET /wvc/v1/forms/{form_key} - Get specific form
        register_rest_route(
            $this->namespace,
            '/forms/(?P<form_key>[a-z_]+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_form'),
                'permission_callback' => '__return_true', // Public endpoint
                'args'                => array(
                    'form_key' => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_key',
                    ),
                ),
            )
        );

        // DELETE /wvc/v1/forms/{form_key} - Delete form
        register_rest_route(
            $this->namespace,
            '/forms/(?P<form_key>[a-z_]+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array($this, 'delete_form'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::MANAGE_OPTIONS_CAP);
                },
                'args'                => array(
                    'form_key' => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_key',
                    ),
                ),
            )
        );
    }

    /**
     * Register a new form or update existing form (UPSERT behavior)
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error on failure.
     */
    public function register_form($request)
    {
        $form_key   = $request->get_param('form_key');
        $form_label = $request->get_param('form_label');
        $fields     = $request->get_param('fields');

        // Check if form already exists
        $existing = get_posts(array(
            'post_type'   => 'wvc_form',
            'meta_key'    => 'wvc_form_key',
            'meta_value'  => $form_key,
            'post_status' => 'publish',
            'numberposts' => 1,
        ));

        if (!empty($existing)) {
            // UPDATE existing form instead of returning error
            $form    = $existing[0];
            $post_id = $form->ID;

            // Update form post
            wp_update_post(array(
                'ID'         => $post_id,
                'post_title' => $form_label,
            ));

            // Update form metadata
            update_post_meta($post_id, 'wvc_form_fields', $fields);

            // Increment form version
            $current_version = get_post_meta($post_id, 'wvc_form_version', true);
            $new_version = !empty($current_version) ? intval($current_version) + 1 : 1;
            update_post_meta($post_id, 'wvc_form_version', $new_version);

            // error_log(sprintf(
            //     '[WVC Form API] Form updated successfully: %s (ID: %d, Label: %s, Fields: %d)',
            //     $form_key,
            //     $post_id,
            //     $form_label,
            //     count($fields)
            // ));

            return rest_ensure_response(array(
                'success' => true,
                'form_id' => $post_id,
                'updated' => true,
                'message' => sprintf(__('Form "%s" updated successfully', 'wvc-theme'), $form_label),
            ));
        }

        // CREATE new form if doesn't exist
        $post_id = wp_insert_post(array(
            'post_type'   => 'wvc_form',
            'post_title'  => $form_label,
            'post_name'   => $form_key,
            'post_status' => 'publish',
        ));

        if (is_wp_error($post_id)) {
            return new WP_Error(
                'form_creation_failed',
                __('Failed to create form', 'wvc-theme'),
                array('status' => 500)
            );
        }

        // Store form metadata
        update_post_meta($post_id, 'wvc_form_key', $form_key);
        update_post_meta($post_id, 'wvc_form_fields', $fields);
        update_post_meta($post_id, 'wvc_form_version', 1);

        // error_log(sprintf(
        //     '[WVC Form API] Form registered successfully: %s (ID: %d, Label: %s, Fields: %d)',
        //     $form_key,
        //     $post_id,
        //     $form_label,
        //     count($fields)
        // ));

        return rest_ensure_response(array(
            'success' => true,
            'form_id' => $post_id,
            'updated' => false,
            'message' => sprintf(__('Form "%s" registered successfully', 'wvc-theme'), $form_label),
        ));
    }

    /**
     * Get all forms
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response Response object.
     */
    public function get_forms($request)
    {
        $forms = get_posts(array(
            'post_type'      => 'wvc_form',
            'posts_per_page' => -1,
            'post_status'    => 'publish',
        ));

        $result = array();
        foreach ($forms as $form) {
            $form_key = get_post_meta($form->ID, 'wvc_form_key', true);
            $fields   = get_post_meta($form->ID, 'wvc_form_fields', true);

            $result[] = array(
                'form_key'   => $form_key,
                'form_label' => $form->post_title,
                'fields'     => $fields ?: array(),
            );
        }

        return rest_ensure_response($result);
    }

    /**
     * Get a specific form
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error on failure.
     */
    public function get_form($request)
    {
        $form_key = $request->get_param('form_key');

        $forms = get_posts(array(
            'post_type'   => 'wvc_form',
            'meta_key'    => 'wvc_form_key',
            'meta_value'  => $form_key,
            'post_status' => 'publish',
            'numberposts' => 1,
        ));

        if (empty($forms)) {
            return new WP_Error(
                'form_not_found',
                __('Form not found', 'wvc-theme'),
                array('status' => 404)
            );
        }

        $form   = $forms[0];
        $fields = get_post_meta($form->ID, 'wvc_form_fields', true);

        return rest_ensure_response(array(
            'form_key'   => $form_key,
            'form_label' => $form->post_title,
            'fields'     => $fields ?: array(),
        ));
    }

    /**
     * Delete a form
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error on failure.
     */
    public function delete_form($request)
    {
        $form_key = $request->get_param('form_key');

        $forms = get_posts(array(
            'post_type'   => 'wvc_form',
            'meta_key'    => 'wvc_form_key',
            'meta_value'  => $form_key,
            'post_status' => 'publish',
            'numberposts' => 1,
        ));

        if (empty($forms)) {
            return new WP_Error(
                'form_not_found',
                __('Form not found', 'wvc-theme'),
                array('status' => 404)
            );
        }

        $form    = $forms[0];
        $deleted = wp_delete_post($form->ID, true);

        if (!$deleted) {
            return new WP_Error(
                'form_deletion_failed',
                __('Failed to delete form', 'wvc-theme'),
                array('status' => 500)
            );
        }

        // error_log(sprintf(
        //     '[WVC Form API] Form deleted successfully: %s (ID: %d)',
        //     $form_key,
        //     $form->ID
        // ));

        return rest_ensure_response(array(
            'success' => true,
            'message' => sprintf(__('Form "%s" deleted successfully', 'wvc-theme'), $form_key),
        ));
    }
}

// Initialize the REST API
function wvc_form_rest_api_init()
{
    $api = new WVC_Form_REST_API();
    add_action('rest_api_init', array($api, 'register_routes'));
}
add_action('init', 'wvc_form_rest_api_init');
