<?php
/**
 *  REST API functionality
 *
 * @package    WVC_Theme
 * @subpackage Menu_REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
/**
 * Class WVC_Menu_REST_API
 *
 * Enables and configures WordPress native menu REST API endpoints
 */
class WVC_Import_REST_API extends WVC_REST_API
{

    public function __construct() {
        parent::__construct();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        register_rest_route(
            $this->namespace,
            '/import',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'import'),
                'permission_callback' => array($this, 'permissions_check'),
            )
        );

        // Save state endpoint
        register_rest_route(
            $this->namespace,
            '/state/save',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'save_state'),
                'permission_callback' => array($this, 'permissions_check'),
            )
        );

        // Add more routes as needed
    }

    /**
     * Import website function
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_Error
     */
    public function import($request)
    {
        // Get the S3 URL from the request
        $s3_url = $request->get_param('s3_url');
        $build_id = $request->get_param('build_id');

        // Validate the URL
        if (empty($s3_url)) {
            return new WP_Error(
                'missing_s3_url',
                __('S3 URL is required', 'wvc-theme'),
                array('status' => 400)
            );
        }

        if ( ! filter_var($s3_url, FILTER_VALIDATE_URL)) {
            return new WP_Error(
                'invalid_s3_url',
                __('Invalid URL format', 'wvc-theme'),
                array('status' => 400)
            );
        }

        if (empty($build_id)) {
            return new WP_Error(
                'missing_build_id',
                __('Build id is required', 'wvc-theme'),
                array('status' => 400)
            );
        }

        try {
            // Include the import class if not already included
            if ( ! class_exists('WVC_Import')) {
                require_once get_template_directory() . '/includes/import/import.php';
            }

            // Initialize the importer
            $importer = new WVC_Import($s3_url, $build_id);

            // Run the import
            $result = $importer->import();

            update_option("wvc_import_result_" . time(), $result);
            if ($result['success']) {
                return rest_ensure_response(array(
                    'success' => true,
                    'message' => $result['message'],
                    'data'    => array(
                        'pages' => $result['pages'],
                        'menus' => $result['menus']
                    )
                ));
            }

            return new WP_Error(
                'import_failed',
                $result['message'],
                array('status' => 500)
            );

        } catch (Exception $e) {
            return new WP_Error(
                'import_exception',
                wvc - rest - api . php__('Import failed: ', 'wvc-theme'),
                array('status' => 500)
            );
        }
    }

    /**
     * Save WVC state
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function save_state($request)
    {
        $action_id          = $request->get_param('action_id');
        $request_id         = $request->get_param('request_id');
        $user_request_short = $request->get_param('user_request_short');

        // Create state entry with timestamp
        $state = array(
            'action_id'          => $action_id,
            'request_id'         => $request_id,
            'user_request_short' => $user_request_short,
            'timestamp'          => current_time('mysql'),
        );

        // Define states directory and file path
        $states_dir = WP_CONTENT_DIR . '/wvc-states';
        $filename   = 'wvc-state.json';
        $filepath   = $states_dir . '/' . $filename;

        // Create directory if it doesn't exist
        if ( ! file_exists($states_dir)) {
            if ( ! wp_mkdir_p($states_dir)) {
                return new WP_Error(
                    'directory_creation_failed',
                    __('Failed to create states directory', 'wvc-theme'),
                    array('status' => 500)
                );
            }
        }

        // Save state to file (overwrites existing file)
        $json_data = wp_json_encode($state, JSON_PRETTY_PRINT);
        $result = file_put_contents($filepath, $json_data);

        if ($result === false) {
            return new WP_Error(
                'save_failed',
                __('Failed to save state to file', 'wvc-theme'),
                array('status' => 500)
            );
        }

        return rest_ensure_response(
            array(
                'success'  => true,
                'filename' => $filename,
                'filepath' => $filepath,
                'state'    => $state,
                'message'  => __('State saved successfully', 'wvc-theme'),
            )
        );
    }
}

// Initialize the REST API
new WVC_Import_REST_API();