<?php
/**
 * Post Operations REST API Class
 *
 * Provides REST API endpoints for creating, retrieving, updating, and deleting WordPress posts,
 * categories, and tags.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/post-manager.php';

/**
 * Class WVC_Post_REST_API
 *
 * Handles post, category, and tag operations via REST API
 */
class WVC_Post_REST_API extends WVC_REST_API {

    /**
     * Post Manager instance
     *
     * @var WVC_Post_Manager
     */
    private $post_manager;

    /**
     * Required capability for posts management
     */
    const EDIT_POSTS_CAP = 'edit_posts';

    /**
     * Required capability for managing categories and tags
     */
    const MANAGE_CATEGORIES_CAP = 'manage_categories';

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        $this->post_manager = WVC_Post_Manager::get_instance();
    }


    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Register post endpoints
        $this->register_post_routes();
        
        // Register category endpoints
        $this->register_category_routes();
        
        // Register tag endpoints
        $this->register_tag_routes();
        
        // Register custom post type endpoints
        $this->register_custom_post_type_routes();
    }

    /**
     * Register post-related routes
     */
    private function register_post_routes() {
        // Create post endpoint
        register_rest_route(
            $this->namespace,
            '/posts',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_post' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(
                    'title'           => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The title of the post', 'wvc-theme' ),
                    ),
                    'content'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The content of the post', 'wvc-theme' ),
                    ),
                    'excerpt'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The excerpt of the post', 'wvc-theme' ),
                    ),
                    'status'          => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'default'           => 'draft',
                        'enum'              => array( 'publish', 'draft', 'pending', 'private' ),
                        'description'       => __( 'The status of the post', 'wvc-theme' ),
                    ),
                    'author'          => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The author ID of the post', 'wvc-theme' ),
                    ),
                    'categories'      => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of category IDs', 'wvc-theme' ),
                    ),
                    'tags'            => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of tag IDs', 'wvc-theme' ),
                    ),
                    'meta'            => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the post', 'wvc-theme' ),
                    ),
                    'featured_image'  => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'Featured image ID for the post', 'wvc-theme' ),
                    ),
                    'comment_status'  => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Comment status for the post', 'wvc-theme' ),
                    ),
                    'ping_status'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Ping status for the post', 'wvc-theme' ),
                    ),
                    'post_type'       => array(
                        'required'    => false,
                        'type'        => 'string',
                        'default'     => 'post',
                        'description' => __( 'The post type (e.g., "post", "testimonial", "product")', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Bulk create/update posts endpoint
        register_rest_route(
            $this->namespace,
            '/posts/bulk',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'bulk_upsert_posts' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(),
            )
        );

        // Get post endpoint
        register_rest_route(
            $this->namespace,
            '/posts/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_post' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the post to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update post endpoint
        register_rest_route(
            $this->namespace,
            '/posts/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_post' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(
                    'id'              => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the post to update', 'wvc-theme' ),
                    ),
                    'title'           => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The title of the post', 'wvc-theme' ),
                    ),
                    'content'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The content of the post', 'wvc-theme' ),
                    ),
                    'excerpt'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The excerpt of the post', 'wvc-theme' ),
                    ),
                    'status'          => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'enum'              => array( 'publish', 'draft', 'pending', 'private' ),
                        'description'       => __( 'The status of the post', 'wvc-theme' ),
                    ),
                    'author'          => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The author ID of the post', 'wvc-theme' ),
                    ),
                    'categories'      => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of category IDs', 'wvc-theme' ),
                    ),
                    'tags'            => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of tag IDs', 'wvc-theme' ),
                    ),
                    'meta'            => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the post', 'wvc-theme' ),
                    ),
                    'featured_image'  => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'Featured image ID for the post', 'wvc-theme' ),
                    ),
                    'comment_status'  => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Comment status for the post', 'wvc-theme' ),
                    ),
                    'ping_status'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Ping status for the post', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Delete post endpoint
        register_rest_route(
            $this->namespace,
            '/posts/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_post' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(
                    'id'    => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the post to delete', 'wvc-theme' ),
                    ),
                    'force' => array(
                        'required'    => false,
                        'type'        => 'boolean',
                        'default'     => false,
                        'description' => __( 'Whether to bypass trash and force deletion', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    /**
     * Register custom post type routes
     */
    private function register_custom_post_type_routes() {
        // Get all registered custom post types
        $cpt_configs = get_option( 'wvc_custom_post_types', array() );

        if ( empty( $cpt_configs ) ) {
            return;
        }

        // Loop through each custom post type and register routes
        foreach ( $cpt_configs as $post_type_key => $config ) {
            // Skip if post type doesn't exist
            if ( ! post_type_exists( $post_type_key ) ) {
                continue;
            }

            // Get schema from config (custom fields only)
            $fields = isset( $config['fields'] ) ? $config['fields'] : array();

            // Build args array starting with standard WordPress post fields
            $args = array(
                'id'       => array(
                    'required'    => false,
                    'type'        => 'integer',
                    'description' => __( 'The ID of the post', 'wvc-theme' ),
                ),
                'title'    => array(
                    'required'          => false,
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                    'description'       => __( 'The title of the post', 'wvc-theme' ),
                ),
                'content'  => array(
                    'required'    => false,
                    'type'        => 'string',
                    'description' => __( 'The content/description of the post', 'wvc-theme' ),
                ),
                'excerpt'  => array(
                    'required'    => false,
                    'type'        => 'string',
                    'description' => __( 'The excerpt of the post', 'wvc-theme' ),
                ),
                'slug'     => array(
                    'required'          => false,
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_title',
                    'description'       => __( 'The slug of the post', 'wvc-theme' ),
                ),
            );

            // Add custom fields from schema
            foreach ( $fields as $field ) {
                $field_key = isset( $field['key'] ) ? $field['key'] : '';
                if ( empty( $field_key ) ) {
                    continue;
                }

                $field_type = isset( $field['type'] ) ? $field['type'] : 'string';

                // Determine WordPress REST API type
                $rest_type = 'string';
                $sanitize_callback = 'sanitize_text_field';
                
                switch ( $field_type ) {
                    case 'number':
                    case 'image':
                    case 'another_custom_post_type':
                        $rest_type = 'integer';
                        $sanitize_callback = 'absint';
                        break;
                    case 'checkbox':
                        $rest_type = 'boolean';
                        $sanitize_callback = 'rest_sanitize_boolean';
                        break;
                    case 'textarea':
                        $sanitize_callback = 'sanitize_textarea_field';
                        break;
                    case 'url':
                        $sanitize_callback = 'esc_url_raw';
                        break;
                    case 'email':
                        $sanitize_callback = 'sanitize_email';
                        break;
                }

                // For READABLE endpoints, all query parameters are optional (used for filtering)
                $args[ $field_key ] = array(
                    'required'          => false,
                    'type'              => $rest_type,
                    'sanitize_callback' => $sanitize_callback,
                    'description'       => isset( $field['label'] ) ? $field['label'] : $field_key,
                );
            }

            // Register route for listing posts: /{post_type}/
            // Use closure to capture post_type_key
            register_rest_route(
                $this->namespace,
                '/wvc_cpt_' . $post_type_key,
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => function( $request ) use ( $post_type_key ) {
                        return $this->get_custom_post_type_posts( $request, $post_type_key );
                    },
                    'permission_callback' => '__return_true', // Public access for reading
                    'args'                => array_merge(
                        array(
                            'per_page' => array(
                                'required'    => false,
                                'type'        => 'integer',
                                'default'     => 10,
                                'description' => __( 'Number of posts to return per page', 'wvc-theme' ),
                            ),
                            'page'     => array(
                                'required'    => false,
                                'type'        => 'integer',
                                'default'     => 1,
                                'description' => __( 'Page number', 'wvc-theme' ),
                            ),
                            'status'   => array(
                                'required'    => false,
                                'type'        => 'string',
                                'default'     => 'publish',
                                'description' => __( 'Post status', 'wvc-theme' ),
                            ),
                        ),
                        $args
                    ),
                )
            );

            // Register route for getting single post by ID: /{post_type}/{id}
            // Use closure to capture post_type_key
            register_rest_route(
                $this->namespace,
                '/wvc_cpt_' . $post_type_key . '/(?P<id>\d+)',
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => function( $request ) use ( $post_type_key ) {
                        return $this->get_custom_post_type_post( $request, $post_type_key );
                    },
                    'permission_callback' => '__return_true', // Public access for reading
                    'args'                => array(
                        'id' => array(
                            'required'    => true,
                            'type'        => 'integer',
                            'description' => __( 'The ID of the post to retrieve', 'wvc-theme' ),
                        ),
                    ),
                )
            );

            // Register route for getting single post by slug: /{post_type}/slug/{slug}
            // Use closure to capture post_type_key
            register_rest_route(
                $this->namespace,
                '/wvc_cpt_' . $post_type_key . '/slug/(?P<slug>[a-zA-Z0-9-]+)',
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => function( $request ) use ( $post_type_key ) {
                        return $this->get_custom_post_type_post_by_slug( $request, $post_type_key );
                    },
                    'permission_callback' => '__return_true', // Public access for reading
                    'args'                => array(
                        'slug' => array(
                            'required'          => true,
                            'type'              => 'string',
                            'sanitize_callback' => 'sanitize_title',
                            'description'       => __( 'The slug of the post to retrieve', 'wvc-theme' ),
                        ),
                    ),
                )
            );
        }
    }

    /**
     * Register category-related routes
     */
    private function register_category_routes() {
        // Create category endpoint
        register_rest_route(
            $this->namespace,
            '/categories',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_category' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'name'        => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the category', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the category', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the category', 'wvc-theme' ),
                    ),
                    'parent'      => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'default'     => 0,
                        'description' => __( 'The parent ID of the category', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the category', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Get category endpoint
        register_rest_route(
            $this->namespace,
            '/categories/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_category' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the category to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update category endpoint
        register_rest_route(
            $this->namespace,
            '/categories/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_category' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id'          => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the category to update', 'wvc-theme' ),
                    ),
                    'name'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the category', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the category', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the category', 'wvc-theme' ),
                    ),
                    'parent'      => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The parent ID of the category', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the category', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Delete category endpoint
        register_rest_route(
            $this->namespace,
            '/categories/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_category' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the category to delete', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    /**
     * Register tag-related routes
     */
    private function register_tag_routes() {
        // Create tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_tag' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'name'        => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the tag', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the tag', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the tag', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the tag', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Get tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_tag' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the tag to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_tag' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id'          => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the tag to update', 'wvc-theme' ),
                    ),
                    'name'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the tag', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the tag', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the tag', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the tag', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Delete tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_tag' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the tag to delete', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    /**
     * Create a new post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_post( $request ) {
        $args = array(
            'title'          => $request->get_param( 'title' ),
            'content'        => $request->get_param( 'content' ),
            'excerpt'        => $request->get_param( 'excerpt' ),
            'status'         => $request->get_param( 'status' ),
            'author'         => $request->get_param( 'author' ),
            'categories'     => $request->get_param( 'categories' ),
            'tags'           => $request->get_param( 'tags' ),
            'meta'           => $request->get_param( 'meta' ),
            'featured_image' => $request->get_param( 'featured_image' ),
            'comment_status' => $request->get_param( 'comment_status' ),
            'ping_status'    => $request->get_param( 'ping_status' ),
            'post_type'      => $request->get_param( 'post_type' ),
        );

        $result = $this->post_manager->create_post( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_post( $request ) {
        $post_id = $request->get_param( 'id' );
        $result  = $this->post_manager->get_post( $post_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_post( $request ) {
        $post_id = $request->get_param( 'id' );
        $args    = array(
            'title'          => $request->get_param( 'title' ),
            'content'        => $request->get_param( 'content' ),
            'excerpt'        => $request->get_param( 'excerpt' ),
            'status'         => $request->get_param( 'status' ),
            'author'         => $request->get_param( 'author' ),
            'categories'     => $request->get_param( 'categories' ),
            'tags'           => $request->get_param( 'tags' ),
            'meta'           => $request->get_param( 'meta' ),
            'featured_image' => $request->get_param( 'featured_image' ),
            'comment_status' => $request->get_param( 'comment_status' ),
            'ping_status'    => $request->get_param( 'ping_status' ),
        );

        $result = $this->post_manager->update_post( $post_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Bulk create/update posts
     *
     * Accepts a JSON payload with either:
     * - Root-level object mapping keys to post data
     * - Envelope object with "posts" key containing the mapping
     *
     * Each post data object can include an "id" field:
     * - If "id" is present and valid, the post will be updated
     * - If "id" is missing or invalid, a new post will be created
     *
     * Returns a mapping of keys to post IDs (0 for failed operations)
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function bulk_upsert_posts( $request ) {
        $payload = $request->get_json_params();

        // Normalize input: accept either a root-level dict, or an envelope with "posts" dict
        $items = array();
        if ( is_array( $payload ) ) {
            if ( array_key_exists( 'posts', $payload ) && is_array( $payload['posts'] ) ) {
                $items = $payload['posts'];
            } else {
                $items = $payload;
            }
        }

        if ( empty( $items ) || ! is_array( $items ) ) {
            return new WP_Error(
                'invalid_body',
                __( 'Request body must be an object mapping keys to post data', 'wvc-theme' ),
                array( 'status' => 400 )
            );
        }

        $ids_by_key = array();

        foreach ( $items as $key => $post_data ) {
            if ( ! is_array( $post_data ) ) {
                $ids_by_key[ (string) $key ] = 0;
                continue;
            }

            $id = null;
            if ( isset( $post_data['id'] ) ) {
                $id = absint( $post_data['id'] );
                unset( $post_data['id'] );
            }

            // Prepare args for post manager
            $args = array(
                'title'          => isset( $post_data['title'] ) ? $post_data['title'] : null,
                'content'        => isset( $post_data['content'] ) ? $post_data['content'] : null,
                'excerpt'        => isset( $post_data['excerpt'] ) ? $post_data['excerpt'] : null,
                'status'         => isset( $post_data['status'] ) ? $post_data['status'] : null,
                'author'         => isset( $post_data['author'] ) ? $post_data['author'] : null,
                'categories'     => isset( $post_data['categories'] ) ? $post_data['categories'] : null,
                'tags'           => isset( $post_data['tags'] ) ? $post_data['tags'] : null,
                'meta'           => isset( $post_data['meta'] ) ? $post_data['meta'] : null,
                'featured_image' => isset( $post_data['featured_image'] ) ? $post_data['featured_image'] : null,
                'comment_status' => isset( $post_data['comment_status'] ) ? $post_data['comment_status'] : null,
                'ping_status'    => isset( $post_data['ping_status'] ) ? $post_data['ping_status'] : null,
                'post_type'      => isset( $post_data['post_type'] ) ? $post_data['post_type'] : null,
            );

            // Remove null values to avoid overwriting with null
            $args = array_filter( $args, function( $value ) {
                return $value !== null;
            } );

            if ( $id ) {
                $result = $this->post_manager->update_post( $id, $args );
                if ( is_wp_error( $result ) ) {
                    $ids_by_key[ (string) $key ] = (int) $id;
                } else {
                    $ids_by_key[ (string) $key ] = ( is_array( $result ) && isset( $result['id'] ) ) ? (int) $result['id'] : (int) $id;
                }
            } else {
                $result = $this->post_manager->create_post( $args );
                if ( is_wp_error( $result ) ) {
                    $ids_by_key[ (string) $key ] = 0;
                } else {
                    $ids_by_key[ (string) $key ] = ( is_array( $result ) && isset( $result['id'] ) ) ? (int) $result['id'] : 0;
                }
            }
        }

        return rest_ensure_response( $ids_by_key );
    }

    /**
     * Delete a post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_post( $request ) {
        $post_id = $request->get_param( 'id' );
        $force   = $request->get_param( 'force' );

        $result = $this->post_manager->delete_post( $post_id, $force );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Create a new category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_category( $request ) {
        $args = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'parent'      => $request->get_param( 'parent' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->create_category( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $result      = $this->post_manager->get_category( $category_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $args        = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'parent'      => $request->get_param( 'parent' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->update_category( $category_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $result      = $this->post_manager->delete_category( $category_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Create a new tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_tag( $request ) {
        $args = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->create_tag( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $result = $this->post_manager->get_tag( $tag_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $args   = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->update_tag( $tag_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $result = $this->post_manager->delete_tag( $tag_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get posts for a custom post type
     *
     * @param WP_REST_Request $request The request object
     * @param string          $post_type The post type
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_custom_post_type_posts( $request, $post_type = null ) {
        // If post_type not provided, try to extract from route
        if ( empty( $post_type ) ) {
            $route = $request->get_route();
            $namespace_pattern = '/' . $this->namespace . '/';
            $route_without_namespace = str_replace( $namespace_pattern, '', $route );
            $route_parts = explode( '/', trim( $route_without_namespace, '/' ) );
            $route_part = ! empty( $route_parts[0] ) ? $route_parts[0] : '';
            // Remove wvc_cpt_ prefix if present
            if ( strpos( $route_part, 'wvc_cpt_' ) === 0 ) {
                $post_type = substr( $route_part, 8 );
            } else {
                $post_type = $route_part;
            }
        }

        // Validate post type exists
        if ( empty( $post_type ) || ! post_type_exists( $post_type ) ) {
            return new WP_Error(
                'invalid_post_type',
                __( 'Invalid post type', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        // Get query parameters
        $per_page = $request->get_param( "per_page" );
        $page = $request->get_param( "page" );
        $status = $request->get_param( "status" );
        $id = $request->get_param( "id" );
        $title = $request->get_param( "title" );
        $content = $request->get_param( "content" );
        $excerpt = $request->get_param( "excerpt" );
        $slug = $request->get_param( "slug" );
        $include = $request->get_param( "include" );

        // Build query args
        $query_args = array(
            "post_type"      => $post_type,
            "posts_per_page" => $per_page ? intval( $per_page ) : 10,
            "paged"          => $page ? intval( $page ) : 1,
            "post_status"    => $status ? sanitize_text_field( $status ) : "publish",
        );

        // Add filtering by standard fields
        if ( ! empty( $id ) ) {
            $query_args["p"] = intval( $id );
        }
        if ( ! empty( $slug ) ) {
            $query_args["name"] = sanitize_title( $slug );
        }
        if ( ! empty( $title ) ) {
            $query_args["s"] = sanitize_text_field( $title );
            $query_args["sentence"] = true; // Exact phrase match
        }
        if ( ! empty( $content ) ) {
            // WordPress doesn't have direct content filtering, but we can use search
            if ( empty( $query_args["s"] ) ) {
                $query_args["s"] = sanitize_text_field( $content );
            }
        }
        if ( ! empty( $excerpt ) ) {
            // WordPress doesn't have direct excerpt filtering, but we can use search
            if ( empty( $query_args["s"] ) ) {
                $query_args["s"] = sanitize_text_field( $excerpt );
            }
        }

        // Support include parameter (similar to core WP REST API)
        if ( ! empty( $include ) ) {
            $include_ids = array();

            if ( is_string( $include ) ) {
                // Handle comma separated string: "1,2,3"
                $include_ids = array_filter(
                    array_map(
                        "intval",
                        explode( ",", $include )
                    )
                );
            } elseif ( is_array( $include ) ) {
                // Handle array: include[]=1&include[]=2
                $include_ids = array_filter(
                    array_map(
                        "intval",
                        $include
                    )
                );
            } else {
                // Fallback single value
                $include_ids = array( intval( $include ) );
            }

            if ( ! empty( $include_ids ) ) {
                $query_args["post__in"] = $include_ids;
                $query_args["orderby"] = "post__in";

                // When include is present, ignore single id / slug filters to avoid conflicts
                unset( $query_args["p"], $query_args["name"] );
            }
        }

        // Get posts
        $posts = get_posts( $query_args );

        // Check if _fields parameter is used
        $fields_param = $request->get_param( '_fields' );
        $use_wp_format = ! empty( $fields_param );

        // Format response
        $formatted_posts = array();
        foreach ( $posts as $post ) {
            $post_data = $this->post_manager->get_post( $post->ID );
            if ( ! is_wp_error( $post_data ) ) {
                // If _fields is used, format to match WordPress REST API format
                if ( $use_wp_format ) {
                    $formatted_posts[] = $this->format_post_for_wp_rest_api( $post_data, $post, $request );
                } else {
                    $formatted_posts[] = $post_data;
                }
            }
        }

        // Get total count for pagination
        $total_query = new WP_Query( array_merge( $query_args, array( 'posts_per_page' => -1 ) ) );
        $total_posts = $total_query->found_posts;
        $total_pages = ceil( $total_posts / $query_args['posts_per_page'] );

        // If _fields is used, return in WordPress REST API format (array of posts)
        if ( $use_wp_format ) {
            return rest_ensure_response( $formatted_posts );
        }

        // Otherwise return our custom format
        return rest_ensure_response( array(
            'success'     => true,
            'data'        => $formatted_posts,
            'pagination' => array(
                'total'       => $total_posts,
                'per_page'    => $query_args['posts_per_page'],
                'current_page' => $query_args['paged'],
                'total_pages' => $total_pages,
            ),
        ) );
    }

    /**
     * Format post data to match WordPress REST API format
     *
     * @param array           $post_data Post data from post manager
     * @param WP_Post         $post WordPress post object
     * @param WP_REST_Request $request The request object
     *
     * @return array Formatted post data
     */
    private function format_post_for_wp_rest_api( $post_data, $post, $request ) {
        $formatted = array(
            'id'    => $post_data['id'],
            'date'  => mysql2date( 'c', $post->post_date, false ),
            'date_gmt' => mysql2date( 'c', $post->post_date_gmt, false ),
            'guid'  => array(
                'rendered' => get_the_guid( $post->ID ),
            ),
            'modified' => mysql2date( 'c', $post->post_modified, false ),
            'modified_gmt' => mysql2date( 'c', $post->post_modified_gmt, false ),
            'slug'  => $post->post_name,
            'status' => $post_data['status'],
            'type'  => $post->post_type,
            'link'  => $post_data['permalink'],
            'title' => array(
                'rendered' => $post_data['title'],
            ),
            'content' => array(
                'rendered' => $post_data['content'],
                'protected' => false,
            ),
            'excerpt' => array(
                'rendered' => $post_data['excerpt'],
                'protected' => false,
            ),
            'author' => $post_data['author'],
            'featured_media' => $post_data['featured_image'] ? $post_data['featured_image'] : 0,
            'comment_status' => $post_data['comment_status'],
            'ping_status' => $post_data['ping_status'],
            'sticky' => is_sticky( $post->ID ),
            'template' => '',
            'format' => get_post_format( $post->ID ) ?: 'standard',
            'meta' => $post_data['meta'],
            'categories' => $post_data['categories'],
            'tags' => $post_data['tags'],
        );

        // Add _links
        $formatted['_links'] = array(
            'self' => array(
                array(
                    'href' => rest_url( $this->namespace . '/wvc_cpt_' . $post->post_type . '/' . $post->ID ),
                ),
            ),
            'collection' => array(
                array(
                    'href' => rest_url( $this->namespace . '/wvc_cpt_' . $post->post_type ),
                ),
            ),
            'about' => array(
                array(
                    'href' => rest_url( 'wp/v2/types/' . $post->post_type ),
                ),
            ),
            'author' => array(
                array(
                    'href' => rest_url( 'wp/v2/users/' . $post_data['author'] ),
                    'embeddable' => true,
                ),
            ),
            'replies' => array(
                array(
                    'href' => add_query_arg( 'post', $post->ID, rest_url( 'wp/v2/comments' ) ),
                    'embeddable' => true,
                ),
            ),
            'version-history' => array(
                array(
                    'href' => rest_url( 'wp/v2/' . $post->post_type . '/' . $post->ID . '/revisions' ),
                ),
            ),
            'predecessor-version' => array(),
            'wp:attachment' => array(
                array(
                    'href' => add_query_arg( 'parent', $post->ID, rest_url( 'wp/v2/media' ) ),
                ),
            ),
            'wp:term' => array(),
            'curies' => array(
                array(
                    'name' => 'wp',
                    'href' => 'https://api.w.org/{rel}',
                    'templated' => true,
                ),
            ),
        );

        // Add featured media link if exists
        if ( ! empty( $post_data['featured_image'] ) ) {
            $formatted['_links']['wp:featuredmedia'] = array(
                array(
                    'href' => rest_url( 'wp/v2/media/' . $post_data['featured_image'] ),
                    'embeddable' => true,
                ),
            );
        }

        // Add category links
        if ( ! empty( $post_data['categories'] ) ) {
            foreach ( $post_data['categories'] as $cat_id ) {
                $formatted['_links']['wp:term'][] = array(
                    'href' => rest_url( 'wp/v2/categories/' . $cat_id ),
                    'embeddable' => true,
                    'taxonomy' => 'category',
                );
            }
        }

        // Add tag links
        if ( ! empty( $post_data['tags'] ) ) {
            foreach ( $post_data['tags'] as $tag_id ) {
                $formatted['_links']['wp:term'][] = array(
                    'href' => rest_url( 'wp/v2/tags/' . $tag_id ),
                    'embeddable' => true,
                    'taxonomy' => 'post_tag',
                );
            }
        }

        // Handle _embed if requested
        $embed = $request->get_param( '_embed' );
        if ( $embed ) {
            $formatted['_embedded'] = array();
            
            // Embed featured media
            if ( ! empty( $post_data['featured_image'] ) ) {
                $attachment = get_post( $post_data['featured_image'] );
                if ( $attachment ) {
                    $formatted['_embedded']['wp:featuredmedia'] = array(
                        array(
                            'id' => $attachment->ID,
                            'date' => mysql2date( 'c', $attachment->post_date, false ),
                            'slug' => $attachment->post_name,
                            'type' => $attachment->post_type,
                            'link' => get_permalink( $attachment->ID ),
                            'title' => array( 'rendered' => $attachment->post_title ),
                            'source_url' => wp_get_attachment_url( $attachment->ID ),
                        ),
                    );
                }
            }
        }

        return $formatted;
    }

    /**
     * Get a single post for a custom post type
     *
     * @param WP_REST_Request $request The request object
     * @param string          $post_type The post type
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_custom_post_type_post( $request, $post_type = null ) {
        $post_id = $request->get_param( 'id' );

        // If post_type not provided, try to extract from route
        if ( empty( $post_type ) ) {
            $route = $request->get_route();
            $namespace_pattern = '/' . $this->namespace . '/';
            $route_without_namespace = str_replace( $namespace_pattern, '', $route );
            $route_parts = explode( '/', trim( $route_without_namespace, '/' ) );
            $route_part = ! empty( $route_parts[0] ) ? $route_parts[0] : '';
            // Remove wvc_cpt_ prefix if present
            if ( strpos( $route_part, 'wvc_cpt_' ) === 0 ) {
                $post_type = substr( $route_part, 8 );
            } else {
                $post_type = $route_part;
            }
        }

        // Validate post type exists
        if ( empty( $post_type ) || ! post_type_exists( $post_type ) ) {
            return new WP_Error(
                'invalid_post_type',
                __( 'Invalid post type', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        // Get the post
        $post = get_post( $post_id );

        // Validate post exists and matches post type
        if ( ! $post || $post->post_type !== $post_type ) {
            return new WP_Error(
                'post_not_found',
                __( 'Post not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        // Get post data using post manager
        $result = $this->post_manager->get_post( $post_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        // Check if _fields parameter is used
        $fields_param = $request->get_param( '_fields' );
        if ( ! empty( $fields_param ) ) {
            // Format to match WordPress REST API format
            $result = $this->format_post_for_wp_rest_api( $result, $post, $request );
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a single post for a custom post type by slug
     *
     * @param WP_REST_Request $request The request object
     * @param string          $post_type The post type
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_custom_post_type_post_by_slug( $request, $post_type = null ) {
        $slug = $request->get_param( 'slug' );

        // If post_type not provided, try to extract from route
        if ( empty( $post_type ) ) {
            $route = $request->get_route();
            $namespace_pattern = '/' . $this->namespace . '/';
            $route_without_namespace = str_replace( $namespace_pattern, '', $route );
            $route_parts = explode( '/', trim( $route_without_namespace, '/' ) );
            $route_part = ! empty( $route_parts[0] ) ? $route_parts[0] : '';
            // Remove wvc_cpt_ prefix if present
            if ( strpos( $route_part, 'wvc_cpt_' ) === 0 ) {
                $post_type = substr( $route_part, 8 );
            } else {
                $post_type = $route_part;
            }
        }

        // Validate post type exists
        if ( empty( $post_type ) || ! post_type_exists( $post_type ) ) {
            return new WP_Error(
                'invalid_post_type',
                __( 'Invalid post type', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        // Get the post by slug
        $posts = get_posts( array(
            'post_type'      => $post_type,
            'name'           => $slug,
            'posts_per_page' => 1,
            'post_status'    => 'publish',
        ) );

        if ( empty( $posts ) ) {
            return new WP_Error(
                'post_not_found',
                __( 'Post not found', 'wvc-theme' ),
                array( 'status' => 404 )
            );
        }

        $post = $posts[0];

        // Get post data using post manager
        $result = $this->post_manager->get_post( $post->ID );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        // Check if _fields parameter is used
        $fields_param = $request->get_param( '_fields' );
        if ( ! empty( $fields_param ) ) {
            // Format to match WordPress REST API format
            $result = $this->format_post_for_wp_rest_api( $result, $post, $request );
        }

        return rest_ensure_response( $result );
    }
}

// Initialize the Post Operations REST API
new WVC_Post_REST_API();