<?php
/**
 * Product Operations REST API Class
 *
 * Provides REST API endpoints for creating, retrieving, updating, and deleting WooCommerce products,
 * product categories, and product tags.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/product-manager.php';

/**
 * Class WVC_Product_REST_API
 *
 * Handles product, category, and tag operations via REST API
 */
class WVC_Product_REST_API extends WVC_REST_API {

    /**
     * Product Manager instance
     *
     * @var WVC_Product_Manager
     */
    private $product_manager;

    /**
     * Required capability for products management
     */
    const EDIT_PRODUCTS_CAP = 'edit_products';

    /**
     * Required capability for managing product categories and tags
     */
    const MANAGE_PRODUCT_CATEGORIES_CAP = 'manage_product_terms';

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        add_filter( "rest_product_query", array( $this, 'query_args' ), 10, 2 );
        add_action( 'rest_api_init', array( $this, 'register_custom_fields' ) );

        $this->product_manager = WVC_Product_Manager::get_instance();
    }


    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Register product endpoints
        $this->register_product_routes();
        
    }

    /**
     * Register product-related routes
     */
    private function register_product_routes() {
        // Create product endpoint
        register_rest_route(
            $this->namespace,
            '/products',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_product' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_PRODUCTS_CAP );
                },
                'args'                => array(
                    'title'               => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'minLength'         => 1,
                        'maxLength'         => 200,
                        'description'       => __( 'Product title', 'wvc-theme' ),
                    ),
                    'excerpt'   => array(
                        'required'          => false,
                        'type'              => 'string',
                        'description'       => __( 'Excerpt of the product', 'wvc-theme' ),
                    ),
                    'content'    => array(
                        'required'          => false,
                        'type'              => 'string',
                        'description'       => __( 'Content of the product', 'wvc-theme' ),
                    ),
                    'price'               => array(
                        'required'          => true,
                        'type'              => 'number',
                        'minimum'           => 0,
                        'multipleOf'        => 0.01,
                        'description'       => __( 'Regular price of the product', 'wvc-theme' ),
                    ),
                    'discounted_price'    => array(
                        'required'          => false,
                        'type'              => array( 'number', 'null' ),
                        'minimum'           => 0,
                        'multipleOf'        => 0.01,
                        'description'       => __( 'Discounted price of the product (null if no discount)', 'wvc-theme' ),
                    ),
                    'categories'          => array(
                        'required'          => false,
                        'type'              => 'array',
                        'items'             => array(
                            'type'    => 'string',
                            'pattern' => '^[a-z0-9-_]+$',
                        ),
                        'description'       => __( 'Array of category slugs this product belongs to', 'wvc-theme' ),
                    ),
                    'tags'                => array(
                        'required'          => false,
                        'type'              => 'array',
                        'items'             => array(
                            'type'    => 'string',
                            'pattern' => '^[a-z0-9-_]+$',
                        ),
                        'description'       => __( 'Array of tag slugs associated with this product', 'wvc-theme' ),
                    ),
                    'featured_image_url'   => array(
                        'required'          => false,
                        'type'              => 'string',
                        'description'       => __( 'Featured image URL for the product', 'wvc-theme' ),
                    ),
                    'link'                => array(
                        'required'          => false,
                        'type'              => 'string',
                        'description'       => __( 'Permalink of the product', 'wvc-theme' ),
                    ),
                    'status'              => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'default'           => 'publish',
                        'enum'              => array( 'publish', 'draft', 'pending', 'private' ),
                        'description'       => __( 'The status of the product', 'wvc-theme' ),
                    ),
					'uid'                 => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'description'       => __( 'External unique identifier saved to meta wvc_uid', 'wvc-theme' ),
					),
                ),
            )
        );

        // List products endpoint
        register_rest_route(
            $this->namespace,
            '/products',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_products' ),
                'permission_callback' => '__return_true',
                'args'                => array_merge(
                    $this->get_products_collection_params(),
                    array(
                        '_fields' => array(
                            'description'       => __( 'Alias of the core _fields query argument.', 'wvc-theme' ),
                            'type'              => 'string',
                            'required'          => false,
                            'sanitize_callback' => 'sanitize_text_field',
                        ),
                    )
                ),
            )
        );

		// Bulk create/update products
		register_rest_route(
			$this->namespace,
			'/products/bulk',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'bulk_upsert_products' ),
				'permission_callback' => function ( WP_REST_Request $request ) {
					return $this->permissions_check( $request, self::EDIT_PRODUCTS_CAP );
				},
				'args'                => array(),
			)
		);

        // Get product endpoint
        register_rest_route(
            $this->namespace,
            '/products/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_product' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the product to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update product endpoint
        register_rest_route(
            $this->namespace,
            '/products/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_product' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_PRODUCTS_CAP );
                },
                'args'                => array(
                    'id'                  => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the product to update', 'wvc-theme' ),
                    ),
                    'title'               => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'minLength'         => 1,
                        'maxLength'         => 200,
                        'description'       => __( 'Product title', 'wvc-theme' ),
                    ),
                    'short_description'   => array(
                        'required'          => false,
                        'type'              => 'string',
                        'minLength'         => 10,
                        'maxLength'         => 500,
                        'description'       => __( 'Brief product description', 'wvc-theme' ),
                    ),
                    'long_description'    => array(
                        'required'          => false,
                        'type'              => 'string',
                        'minLength'         => 50,
                        'maxLength'         => 5000,
                        'description'       => __( 'Detailed product description', 'wvc-theme' ),
                    ),
                    'price'               => array(
                        'required'          => false,
                        'type'              => 'number',
                        'minimum'           => 0,
                        'multipleOf'        => 0.01,
                        'description'       => __( 'Regular price of the product', 'wvc-theme' ),
                    ),
                    'discounted_price'    => array(
                        'required'          => false,
                        'type'              => array( 'number', 'null' ),
                        'minimum'           => 0,
                        'multipleOf'        => 0.01,
                        'description'       => __( 'Discounted price of the product (null if no discount)', 'wvc-theme' ),
                    ),
                    'categories'          => array(
                        'required'          => false,
                        'type'              => 'array',
                        'minItems'          => 1,
                        'items'             => array(
                            'type'    => 'string',
                            'pattern' => '^[a-z0-9-_]+$',
                        ),
                        'description'       => __( 'Array of category slugs this product belongs to', 'wvc-theme' ),
                    ),
                    'tags'                => array(
                        'required'          => false,
                        'type'              => 'array',
                        'items'             => array(
                            'type'    => 'string',
                            'pattern' => '^[a-z0-9-_]+$',
                        ),
                        'description'       => __( 'Array of tag slugs associated with this product', 'wvc-theme' ),
                    ),
                    'image_description'   => array(
                        'required'          => false,
                        'type'              => 'string',
                        'minLength'         => 10,
                        'maxLength'         => 1000,
                        'description'       => __( 'Description of the product image for accessibility and AI generation', 'wvc-theme' ),
                    ),
                    'status'              => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'enum'              => array( 'publish', 'draft', 'pending', 'private' ),
                        'description'       => __( 'The status of the product', 'wvc-theme' ),
                    ),
					'uid'                 => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'description'       => __( 'External unique identifier saved to meta wvc_uid', 'wvc-theme' ),
					),
                ),
            )
        );

        // Delete product endpoint
        register_rest_route(
            $this->namespace,
            '/products/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_product' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_PRODUCTS_CAP );
                },
                'args'                => array(
                    'id'    => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the product to delete', 'wvc-theme' ),
                    ),
                    'force' => array(
                        'required'    => false,
                        'type'        => 'boolean',
                        'default'     => false,
                        'description' => __( 'Whether to bypass trash and force deletion', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    

    /**
     * Create a new product
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_product( $request ) {
        $args = array(
            'title'             => $request->get_param( 'title' ),
            'excerpt' => $request->get_param( 'excerpt' ),
            'content'  => $request->get_param( 'content' ),
            'price'             => $request->get_param( 'price' ),
            'discounted_price'  => $request->get_param( 'discounted_price' ),
            'categories'        => $request->get_param( 'categories' ),
            'tags'              => $request->get_param( 'tags' ),
            'image_description' => $request->get_param( 'image_description' ),
            'status'            => $request->get_param( 'status' ),
        );

        $result = $this->product_manager->create_product( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a collection of products using the core posts controller.
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_products( $request ) {
        if ( ! class_exists( 'WP_REST_Posts_Controller' ) ) {
            return new WP_Error(
                'wvc_missing_posts_controller',
                __( 'The core posts controller is unavailable.', 'wvc-theme' ),
                array( 'status' => 500 )
            );
        }

        $fields = $request->get_param( '_fields' );

        if ( ! empty( $fields ) ) {
            $request->set_param( '_fields', $fields );
        }

        $controller = new WP_REST_Posts_Controller( 'product' );

        return $controller->get_items( $request );
    }

    /**
     * Get a product
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_product( $request ) {
        $product_id = $request->get_param( 'id' );
        $result     = $this->product_manager->get_product( $product_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a product
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_product( $request ) {
        $product_id = $request->get_param( 'id' );
        $args       = array(
            'title'             => $request->get_param( 'title' ),
            'short_description' => $request->get_param( 'short_description' ),
            'long_description'  => $request->get_param( 'long_description' ),
            'price'             => $request->get_param( 'price' ),
            'discounted_price'  => $request->get_param( 'discounted_price' ),
            'categories'        => $request->get_param( 'categories' ),
            'tags'              => $request->get_param( 'tags' ),
            'image_description' => $request->get_param( 'image_description' ),
            'status'            => $request->get_param( 'status' ),
        );

        $result = $this->product_manager->update_product( $product_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a product
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_product( $request ) {
        $product_id = $request->get_param( 'id' );
        $force      = $request->get_param( 'force' );

        $result = $this->product_manager->delete_product( $product_id, $force );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

	/**
	 * Bulk create or update products
	 *
	 * Accepts a JSON object (dictionary) where each key maps to a product data object.
	 * Each product object may include an "id". If an id is present, the product
	 * is updated; otherwise, a new product is created. Returns a dictionary mapping
	 * the same input keys to resulting product IDs.
	 *
	 * Also accepts an envelope object with a top-level "products" dictionary.
	 *
	 * Error handling:
	 * - Create error: returns 0 for that key
	 * - Update error: returns the same input id for that key
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function bulk_upsert_products( $request ) {
		$payload = $request->get_json_params();

		// Normalize input: accept either a root-level dict, or an envelope with "products" dict
		$items = array();
		if ( is_array( $payload ) ) {
			if ( array_key_exists( 'products', $payload ) && is_array( $payload['products'] ) ) {
				$items = $payload['products'];
			} else {
				$items = $payload;
			}
		}

		if ( empty( $items ) || ! is_array( $items ) ) {
			return new WP_Error(
				'invalid_body',
				__( 'Request body must be an object mapping keys to product data', 'wvc-theme' ),
				array( 'status' => 400 )
			);
		}

		$ids_by_key = array();

		foreach ( $items as $key => $product_data ) {
			if ( ! is_array( $product_data ) ) {
				$ids_by_key[ (string) $key ] = 0;
				continue;
			}

			$id = null;
			if ( isset( $product_data['id'] ) ) {
				$id = absint( $product_data['id'] );
				unset( $product_data['id'] );
			}

			if ( $id ) {
				$result = $this->product_manager->update_product( $id, $product_data );
				if ( is_wp_error( $result ) ) {
					$ids_by_key[ (string) $key ] = (int) $id;
				} else {
					$ids_by_key[ (string) $key ] = isset( $result['id'] ) ? (int) $result['id'] : (int) $id;
				}
			} else {
				$result = $this->product_manager->create_product( $product_data );
				if ( is_wp_error( $result ) ) {
					$ids_by_key[ (string) $key ] = 0;
				} else {
					$ids_by_key[ (string) $key ] = isset( $result['id'] ) ? (int) $result['id'] : 0;
				}
			}
		}

		return rest_ensure_response( $ids_by_key );
	}

    /**
     * Create a new category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_category( $request ) {
        $args = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'parent'      => $request->get_param( 'parent' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->product_manager->create_category( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $result      = $this->product_manager->get_category( $category_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $args        = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'parent'      => $request->get_param( 'parent' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->product_manager->update_category( $category_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $result      = $this->product_manager->delete_category( $category_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Create a new tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_tag( $request ) {
        $args = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->product_manager->create_tag( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $result = $this->product_manager->get_tag( $tag_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $args   = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->product_manager->update_tag( $tag_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $result = $this->product_manager->delete_tag( $tag_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Register custom REST API fields for products
     */
    public function register_custom_fields() {
        // Register short_description field
        register_rest_field(
            'product',
            'short_description',
            array(
                'get_callback' => function( $post ) {
                    $product = wc_get_product( $post['id'] );
                    return $product ? $product->get_short_description() : '';
                },
                'schema' => array(
                    'description' => __( 'Short description of the product', 'wvc-theme' ),
                    'type'        => 'string',
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register long_description field
        register_rest_field(
            'product',
            'long_description',
            array(
                'get_callback' => function( $post ) {
                    $product = wc_get_product( $post['id'] );
                    return $product ? $product->get_description() : '';
                },
                'schema' => array(
                    'description' => __( 'Long description of the product', 'wvc-theme' ),
                    'type'        => 'string',
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register price field
        register_rest_field(
            'product',
            'price',
            array(
                'get_callback' => function( $post ) {
                    $product = wc_get_product( $post['id'] );
                    return $product ? (float) $product->get_regular_price() : 0;
                },
                'schema' => array(
                    'description' => __( 'Regular price of the product', 'wvc-theme' ),
                    'type'        => 'number',
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register discounted_price field
        register_rest_field(
            'product',
            'discounted_price',
            array(
                'get_callback' => function( $post ) {
                    $product = wc_get_product( $post['id'] );
                    $sale_price = $product ? $product->get_sale_price() : '';
                    return $sale_price ? (float) $sale_price : null;
                },
                'schema' => array(
                    'description' => __( 'Discounted price of the product', 'wvc-theme' ),
                    'type'        => array( 'number', 'null' ),
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register categories field
        register_rest_field(
            'product',
            'categories',
            array(
                'get_callback' => function( $post ) {
                    $product = wc_get_product( $post['id'] );
                    if ( ! $product ) {
                        return array();
                    }
                    $category_ids = $product->get_category_ids();
                    $categories = array();
                    foreach ( $category_ids as $cat_id ) {
                        $term = get_term( $cat_id, 'product_cat' );
                        if ( $term && ! is_wp_error( $term ) ) {
                            $categories[] = array(
                                'id'   => $term->term_id,
                                'name' => $term->name,
                                'slug' => $term->slug,
                            );
                        }
                    }
                    return $categories;
                },
                'schema' => array(
                    'description' => __( 'Product categories', 'wvc-theme' ),
                    'type'        => 'array',
                    'items'       => array(
                        'type'       => 'object',
                        'properties' => array(
                            'id'   => array( 'type' => 'integer' ),
                            'name' => array( 'type' => 'string' ),
                            'slug' => array( 'type' => 'string' ),
                        ),
                    ),
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register tags field
        register_rest_field(
            'product',
            'tags',
            array(
                'get_callback' => function( $post ) {
                    $product = wc_get_product( $post['id'] );
                    if ( ! $product ) {
                        return array();
                    }
                    $tag_ids = $product->get_tag_ids();
                    $tags = array();
                    foreach ( $tag_ids as $tag_id ) {
                        $term = get_term( $tag_id, 'product_tag' );
                        if ( $term && ! is_wp_error( $term ) ) {
                            $tags[] = array(
                                'id'   => $term->term_id,
                                'name' => $term->name,
                                'slug' => $term->slug,
                            );
                        }
                    }
                    return $tags;
                },
                'schema' => array(
                    'description' => __( 'Product tags', 'wvc-theme' ),
                    'type'        => 'array',
                    'items'       => array(
                        'type'       => 'object',
                        'properties' => array(
                            'id'   => array( 'type' => 'integer' ),
                            'name' => array( 'type' => 'string' ),
                            'slug' => array( 'type' => 'string' ),
                        ),
                    ),
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register featured_image_url field
        register_rest_field(
            'product',
            'featured_image_url',
            array(
                'get_callback' => function( $post ) {
                    $url = get_the_post_thumbnail_url( $post['id'], 'full' );

                    return $url ? $url : '';
                },
                'schema' => array(
                    'description' => __( 'Absolute URL of the product featured image', 'wvc-theme' ),
                    'type'        => 'string',
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );

        // Register link field
        register_rest_field(
            'product',
            'link',
            array(
                'get_callback' => function( $post ) {
                    return get_the_permalink( $post['id'] );
                },
                'schema' => array(
                    'description' => __( 'Permalink of the product', 'wvc-theme' ),
                    'type'        => 'string',
                    'context'     => array( 'view', 'edit' ),
                ),
            )
        );
    }

    /**
     * Get the query args for the products collection.
     *
     * @return array
     */
    private function get_products_collection_params() {
        if ( ! class_exists( 'WP_REST_Posts_Controller' ) ) {
            return array();
        }

        $controller = new WP_REST_Posts_Controller( 'product' );

        return $controller->get_collection_params();
    }

    public function query_args( $args, $request ) {
		// Set post_status.
		$args['post_status'] = $request->get_param( 'status' );

		// Parse _fields parameter to determine which fields are requested for response
		$fields_param = $request->get_param( '_fields' );
		$requested_fields = array();
		
		if ( ! empty( $fields_param ) ) {
			// Parse comma-separated fields string into array
			$fields_array = array_map( 'trim', explode( ',', $fields_param ) );
			
			// Extract base field names (remove dot notation like "title.rendered")
			foreach ( $fields_array as $field ) {
				$base_field = explode( '.', $field )[0];
				$requested_fields[] = $base_field;
			}
			
			$requested_fields = array_unique( $requested_fields );
		}

		// Taxonomy query to filter products.
		// Note: prepare_tax_query() in WP_REST_Posts_Controller already handles
		// taxonomies with show_in_rest=true when passed as arrays.
		// We handle mapped parameters, comma-separated strings, and special cases here.
		$tax_query = isset( $args['tax_query'] ) ? $args['tax_query'] : array();
		
		// Check which taxonomies are already in tax_query to avoid duplicates
		$existing_taxonomies = array();
		if ( ! empty( $tax_query ) ) {
			foreach ( $tax_query as $query ) {
				if ( isset( $query['taxonomy'] ) ) {
					$existing_taxonomies[] = $query['taxonomy'];
				}
			}
		}

		// Map between taxonomy name and arg's key (for convenience aliases).
		$taxonomies = array(
			'product_cat'            => 'category',
			'product_tag'            => 'tag',
			'product_shipping_class' => 'shipping_class',
		);

		// Set tax_query for mapped parameters (category, tag, shipping_class).
		// These are convenience aliases that may not match the taxonomy's rest_base.
		foreach ( $taxonomies as $taxonomy => $key ) {
			// Skip if already handled by prepare_tax_query
			if ( in_array( $taxonomy, $existing_taxonomies, true ) ) {
				continue;
			}
			
			if ( ! empty( $request[ $key ] ) && is_array( $request[ $key ] ) ) {
				$request[ $key ] = array_filter( $request[ $key ] );
			}

			if ( ! empty( $request[ $key ] ) ) {
				$tax_query[] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'term_id',
					'terms'    => $request[ $key ],
				);
			}
		}

		// Handle direct taxonomy parameters with comma-separated strings
		// (e.g., product_cat=1,2,3&product_tag=4,5,6)
		// prepare_tax_query handles arrays, but may not handle comma-separated strings
		$direct_taxonomies = array( 'product_cat', 'product_tag', 'product_shipping_class', 'product_type' );
		foreach ( $direct_taxonomies as $taxonomy ) {
			// Skip if already handled by prepare_tax_query
			if ( in_array( $taxonomy, $existing_taxonomies, true ) ) {
				continue;
			}
			
			$tax_param = $request->get_param( $taxonomy );
			if ( ! empty( $tax_param ) ) {
				// If it's already an array, prepare_tax_query likely handled it
				if ( is_array( $tax_param ) ) {
					continue;
				}
				
				// Parse comma-separated string values
				if ( is_string( $tax_param ) ) {
					$terms = array_map( 'trim', explode( ',', $tax_param ) );
					$terms = array_filter( $terms );
				} else {
					$terms = array( $tax_param );
				}

				if ( ! empty( $terms ) ) {
					// Determine field type: 'term_id' for numeric values, 'slug' for product_type
					$field = ( 'product_type' === $taxonomy ) ? 'slug' : 'term_id';
					
					// Convert to integers if using term_id
					if ( 'term_id' === $field ) {
						$terms = array_map( 'absint', $terms );
						$terms = array_filter( $terms );
					}

					if ( ! empty( $terms ) ) {
						$tax_query[] = array(
							'taxonomy' => $taxonomy,
							'field'    => $field,
							'terms'    => $terms,
						);
					}
				}
			}
		}

		// Filter product type by slug (special case - product_type may not have show_in_rest).
		if ( ! empty( $request['type'] ) && ! in_array( 'product_type', $existing_taxonomies, true ) ) {
			$tax_query[] = array(
				'taxonomy' => 'product_type',
				'field'    => 'slug',
				'terms'    => $request['type'],
			);
		}

		// Filter by attribute and term.
		if ( ! empty( $request['attribute'] ) && ! empty( $request['attribute_term'] ) ) {
			if ( in_array( $request['attribute'], wc_get_attribute_taxonomy_names(), true ) ) {
				$tax_query[] = array(
					'taxonomy' => $request['attribute'],
					'field'    => 'term_id',
					'terms'    => $request['attribute_term'],
				);
			}
		}

		// Ensure taxonomy cache is updated if taxonomy fields are requested in _fields
		$needs_term_cache = false;
		if ( ! empty( $requested_fields ) ) {
			$taxonomy_fields = array( 'categories', 'tags' );
			foreach ( $taxonomy_fields as $tax_field ) {
				if ( in_array( $tax_field, $requested_fields, true ) ) {
					$needs_term_cache = true;
					break;
				}
			}
		}

		if ( $needs_term_cache ) {
			$args['update_post_term_cache'] = true;
		}

		if ( ! empty( $tax_query ) ) {
			$args['tax_query'] = $tax_query;
		}

		// Meta query for filtering (not based on _fields, but on actual filter params)
		$meta_query = isset( $args['meta_query'] ) ? $args['meta_query'] : array();

		// Ensure meta cache is updated if meta fields are requested in _fields
		$needs_meta_cache = false;
		if ( ! empty( $requested_fields ) ) {
			$meta_fields = array( 'price', 'discounted_price', 'short_description', 'long_description' );
			foreach ( $meta_fields as $meta_field ) {
				if ( in_array( $meta_field, $requested_fields, true ) ) {
					$needs_meta_cache = true;
					break;
				}
			}
		}

		if ( $needs_meta_cache ) {
			$args['update_post_meta_cache'] = true;
		}

		if ( ! empty( $meta_query ) ) {
			$args['meta_query'] = $meta_query;
		}

		// Apply all WP_Query filters again.
		if ( is_array( $request['filter'] ) ) {
			$args = array_merge( $args, $request['filter'] );
			unset( $args['filter'] );
		}

		// Force the post_type argument, since it's not a user input variable.
		if ( ! empty( $request['sku'] ) ) {
			$args['post_type'] = array( 'product', 'product_variation' );
		} else {
			$args['post_type'] = "product";
		}

		return $args;
	}
}

// Initialize the Product Operations REST API
new WVC_Product_REST_API();