<?php
/**
 * WVC Settings REST API Class
 *
 * Provides REST API endpoints for retrieving and updating WordPress general settings.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';

/**
 * Class WVC_Settings_REST_API
 *
 * Handles WordPress general settings operations via REST API
 */
class WVC_Settings_REST_API extends WVC_REST_API {

    /**
     * Required capability for managing settings
     */
    const MANAGE_OPTIONS_CAP = 'manage_options';

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
    }

    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Get all settings endpoint
        register_rest_route(
            $this->namespace,
            '/settings',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_settings'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::MANAGE_OPTIONS_CAP);
                }
            )
        );

        // Update settings endpoint
        register_rest_route(
            $this->namespace,
            '/settings',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'update_settings'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::MANAGE_OPTIONS_CAP);
                },
                'args'                => array(
                    'settings' => array(
                        'required'    => true,
                        'type'        => 'object',
                        'description' => __('Key-value pairs of settings to update', 'wvc-theme'),
                    )
                )
            )
        );
    }

    /**
     * Get all general WordPress settings
     *
     * @param WP_REST_Request $request The request object
     * @return WP_REST_Response
     */
    public function get_settings($request) {
        $settings = $this->get_general_settings();
        return rest_ensure_response($settings);
    }

    /**
     * Update WordPress settings
     *
     * @param WP_REST_Request $request The request object
     * @return WP_REST_Response
     */
    public function update_settings($request) {
        $result = [
            'success'   => true,
            'updated'   => [],
            'failed'    => [],
            'message'   => '',
        ];

        $settings = $request->get_param('settings');
        
        if (empty($settings) || !is_array($settings)) {
            return new WP_Error(
                'invalid_settings',
                __('Invalid settings data provided', 'wvc-theme'),
                ['status' => 400]
            );
        }

        // List of allowed settings to update
        $allowed_settings = $this->get_allowed_settings();
        
        foreach ($settings as $setting_key => $setting_value) {
            // Check if this is an allowed setting
            if (!in_array($setting_key, $allowed_settings)) {
                $result['failed'][] = [
                    'key'     => $setting_key,
                    'value'   => $setting_value,
                    'message' => 'Setting is not allowed to be updated'
                ];
                continue;
            }

            // Update the setting
            $updated = $this->update_setting($setting_key, $setting_value);
            
            if ($updated) {
                $result['updated'][] = [
                    'key'   => $setting_key,
                    'value' => $setting_value
                ];
            } else {
                $result['failed'][] = [
                    'key'     => $setting_key,
                    'value'   => $setting_value,
                    'message' => 'Failed to update setting'
                ];
            }
        }

        if (count($result['updated']) > 0) {
            $result['message'] = __('Settings updated successfully', 'wvc-theme');
        } else if (count($result['failed']) > 0) {
            $result['message'] = __('Failed to update all settings', 'wvc-theme');
            $result['success'] = false;
        }

        return rest_ensure_response($result);
    }

    /**
     * Get list of general WordPress settings
     *
     * @return array Settings with their values
     */
    private function get_general_settings() {
        $settings = [
            // General
            'blogname'           => get_option('blogname'),
            'blogdescription'    => get_option('blogdescription'),
            'siteurl'            => get_option('siteurl'),
            'home'               => get_option('home'),
            'admin_email'        => get_option('admin_email'),
            'users_can_register' => get_option('users_can_register'),
            'default_role'       => get_option('default_role'),
            'timezone_string'    => get_option('timezone_string'),
            'date_format'        => get_option('date_format'),
            'time_format'        => get_option('time_format'),
            'start_of_week'      => get_option('start_of_week'),
            
            // Reading
            'show_on_front'      => get_option('show_on_front'),
            'page_on_front'      => get_option('page_on_front'),
            'page_for_posts'     => get_option('page_for_posts'),
            'posts_per_page'     => get_option('posts_per_page'),
            'posts_per_rss'      => get_option('posts_per_rss'),
            'rss_use_excerpt'    => get_option('rss_use_excerpt'),
            
            // Discussion
            'default_pingback_flag'       => get_option('default_pingback_flag'),
            'default_ping_status'         => get_option('default_ping_status'),
            'default_comment_status'      => get_option('default_comment_status'),
            'require_name_email'          => get_option('require_name_email'),
            'comment_registration'        => get_option('comment_registration'),
            'close_comments_for_old_posts' => get_option('close_comments_for_old_posts'),
            'close_comments_days_old'     => get_option('close_comments_days_old'),
            'thread_comments'             => get_option('thread_comments'),
            'thread_comments_depth'       => get_option('thread_comments_depth'),
            'page_comments'               => get_option('page_comments'),
            'comments_per_page'           => get_option('comments_per_page'),
            'default_comments_page'       => get_option('default_comments_page'),
            'comment_order'               => get_option('comment_order'),
            'comments_notify'             => get_option('comments_notify'),
            'moderation_notify'           => get_option('moderation_notify'),
            'comment_moderation'          => get_option('comment_moderation'),
            'comment_previously_approved' => get_option('comment_previously_approved'),
            
            // Media
            'thumbnail_size_w'            => get_option('thumbnail_size_w'),
            'thumbnail_size_h'            => get_option('thumbnail_size_h'),
            'thumbnail_crop'              => get_option('thumbnail_crop'),
            'medium_size_w'               => get_option('medium_size_w'),
            'medium_size_h'               => get_option('medium_size_h'),
            'large_size_w'                => get_option('large_size_w'),
            'large_size_h'                => get_option('large_size_h'),
            'uploads_use_yearmonth_folders' => get_option('uploads_use_yearmonth_folders'),
            
            // Permalinks
            'permalink_structure'         => get_option('permalink_structure'),
            'category_base'               => get_option('category_base'),
            'tag_base'                    => get_option('tag_base'),
        ];

        return $settings;
    }

    /**
     * Get list of settings that are allowed to be updated
     *
     * @return array Array of allowed setting keys
     */
    private function get_allowed_settings() {
        return [
            'blogname',
            'blogdescription',
            'admin_email',
            'users_can_register',
            'default_role',
            'timezone_string',
            'date_format',
            'time_format',
            'start_of_week',
            'show_on_front',
            'page_on_front',
            'page_for_posts',
            'posts_per_page',
            'posts_per_rss',
            'rss_use_excerpt',
            'default_pingback_flag',
            'default_ping_status',
            'default_comment_status',
            'require_name_email',
            'comment_registration',
            'close_comments_for_old_posts',
            'close_comments_days_old',
            'thread_comments',
            'thread_comments_depth',
            'page_comments',
            'comments_per_page',
            'default_comments_page',
            'comment_order',
            'comments_notify',
            'moderation_notify',
            'comment_moderation',
            'comment_previously_approved',
            'thumbnail_size_w',
            'thumbnail_size_h',
            'thumbnail_crop',
            'medium_size_w',
            'medium_size_h',
            'large_size_w',
            'large_size_h',
            'uploads_use_yearmonth_folders',
            'permalink_structure',
            'category_base',
            'tag_base',
        ];
    }

    /**
     * Update a specific WordPress setting
     *
     * @param string $key The setting key
     * @param mixed $value The new value
     * @return bool True on success, false on failure
     */
    private function update_setting($key, $value) {
        // Handle special cases
        switch ($key) {
            case 'permalink_structure':
                // When updating permalink structure, flush rewrite rules
                $success = update_option($key, $value);
                flush_rewrite_rules();
                return $success;
                
            case 'category_base':
            case 'tag_base':
                // When updating category/tag bases, flush rewrite rules
                $success = update_option($key, $value);
                flush_rewrite_rules();
                return $success;
                
            case 'admin_email':
                // Validate email
                if (!is_email($value)) {
                    return false;
                }
                return update_option($key, $value);
                
            case 'show_on_front':
                // Validate show_on_front value
                if (!in_array($value, ['posts', 'page'])) {
                    return false;
                }
                return update_option($key, $value);
                
            case 'page_on_front':
            case 'page_for_posts':
                // Validate page IDs
                if ($value && !get_post($value)) {
                    return false;
                }
                return update_option($key, $value);
                
            default:
                // Handle standard options
                return update_option($key, $value);
        }
    }
}

// Initialize the Settings REST API
new WVC_Settings_REST_API();