<?php

namespace WVC\Includes\Sync;

/**
 * WVC Sync
 *
 * This file contains the functionality for syncing wp actions,
 * like page, menu delete to wvc service db
 *
 * @package    WVC_Theme
 * @subpackage Includes\Sync
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Theme Sync Class
 *
 * Handles sync functionality with WVC service
 */
class Sync
{
    /**
     * Initialize hooks for sync functionality
     */
    public static function init()
    {
        self::add_menu_hooks();
        self::add_page_hooks();
    }

    /**
     * Add hooks for menu operations
     */
    private static function add_menu_hooks()
    {
        add_action('wp_delete_nav_menu', array(__CLASS__, 'on_menu_delete'), 10, 1);
        add_action('before_delete_post', array(__CLASS__, 'on_menu_item_delete'), 10, 1);
    }

    /**
     * Add hooks for page operations
     */
    private static function add_page_hooks()
    {
        add_action('before_delete_post', array(__CLASS__, 'on_page_delete'), 10, 1);
    }

    /**
     * Handle menu deletion sync
     *
     * @param int $menu_id The ID of the deleted menu
     */
    public static function on_menu_delete($menu_id)
    {
        if (get_transient("wvc_reset_started")) {
            return;
        }
        if ( ! $menu_id) {
            return;
        }

        $delete_data = array(
            'menu_id'   => $menu_id,
            'timestamp' => current_time('timestamp'),
            'user_id'   => get_current_user_id()
        );

        self::make_sync_request('menu', 'delete', $delete_data);

        do_action('wvc_sync_menu_delete', $menu_id, $delete_data);

        error_log(sprintf(
            'WVC Sync: Menu delete - ID: %d, Data: %s',
            $menu_id,
            wp_json_encode($delete_data)
        ));
    }


    /**
     * Handle menu item deletion sync
     *
     * @param int $post_id The ID of the deleted menu item
     */
    public static function on_menu_item_delete($post_id)
    {
        if (get_transient("wvc_reset_started")) {
            return;
        }
        if ( ! $post_id) {
            return;
        }

        $post = get_post($post_id);
        if ( ! $post || $post->post_type !== 'nav_menu_item') {
            return;
        }

        $menu_id = wp_get_post_terms($post_id, 'nav_menu', array('fields' => 'ids'));
        $menu_id = ! empty($menu_id) ? $menu_id[0] : null;

        $delete_data = array(
            'menu_item_id' => $post_id,
            'menu_id'      => $menu_id,
            'item_title'   => $post->post_title,
            'timestamp'    => current_time('timestamp'),
            'user_id'      => get_current_user_id()
        );

        self::make_sync_request('menu', 'item_delete', $delete_data);

        do_action('wvc_sync_menu_item_delete', $post_id, $delete_data);

        error_log(sprintf(
            'WVC Sync: Menu item delete - ID: %d, Data: %s',
            $post_id,
            wp_json_encode($delete_data)
        ));
    }


    /**
     * Handle page deletion sync
     *
     * @param int $page_id The ID of the deleted page
     */
    public static function on_page_delete($page_id)
    {
        if (get_transient("wvc_reset_started")) {
            return;
        }
        if ( ! $page_id) {
            return;
        }

        $page = get_post($page_id);
        if ( ! $page || ! in_array($page->post_type, array('page', 'revision'))) {
            return;
        }

        // If it's a revision, get the parent page
        if ($page->post_type === 'revision') {
            $parent_id   = $page->post_parent;
            $parent_page = get_post($parent_id);
            if ( ! $parent_page || $parent_page->post_type !== 'page') {
                return;
            }
            // Use parent page data for sync
            $page    = $parent_page;
            $page_id = $parent_id;
        }

        $is_wvc = get_post_meta($page_id, '_wvc_page_url', true);

        if ( ! $is_wvc) {
            // return;
        }

        $delete_data = array(
            'page_id'     => $page_id,
            'page_title'  => $page->post_title,
            'page_slug'   => $page->post_name,
            'page_status' => $page->post_status,
            'timestamp'   => current_time('timestamp'),
            'user_id'     => get_current_user_id()
        );

        self::make_sync_request('page', 'delete', $delete_data);

        do_action('wvc_sync_page_delete', $page_id, $delete_data);

        error_log(sprintf(
            'WVC Sync: Page delete - ID: %d, Data: %s',
            $page_id,
            wp_json_encode($delete_data)
        ));
    }

    /**
     * Make HTTP request to sync endpoint
     *
     * @param string $source Source type ('menu' or 'page')
     * @param string $action Action type ('edit' or 'delete')
     * @param array $data Data to sync
     *
     * @return array|WP_Error Response array or WP_Error on failure
     */
    private static function make_sync_request($source, $action, $data)
    {
        // Get required configuration
        $base_url     = defined('TENWEB_WVC_URL') ? TENWEB_WVC_URL : '';
        $workspace_id = get_site_option('tenweb_workspace_id');
        $domain_id    = get_site_option('tenweb_domain_id');
        $access_token = get_site_option('tenweb_access_token');
        // Validate configuration
        if (empty($base_url) || empty($workspace_id) || empty($domain_id) || empty($access_token)) {
            error_log('WVC Sync: Missing required configuration - base_url, workspace_id, domain_id, or access_token');

            return new \WP_Error(
                'missing_config',
                'Missing required sync configuration',
                array('status' => 400)
            );
        }

        // Build endpoint URL
        $endpoint_url = rtrim($base_url, '/') . "/api/workspaces/{$workspace_id}/domains/{$domain_id}/wp-sync/";

        // Prepare request headers
        $headers = array(
            'Authorization' => 'Bearer ' . $access_token,
            'Accept'        => 'application/x.10webwvc.v1+json',
            'Content-Type'  => 'application/json',
        );

        // Structure the request body
        $request_body = array(
            'source' => $source,
            'action' => $action,
            'data'   => $data
        );

        // Prepare request arguments
        $args = array(
            'method'    => 'POST',
            'headers'   => $headers,
            'body'      => wp_json_encode($request_body),
            'timeout'   => 30,
            'blocking'  => false, // Non-blocking request to avoid slowing down WordPress operations
            'sslverify' => true,
        );

        // Make the request
        $response = wp_remote_post($endpoint_url, $args);

        // Log the request for debugging
        error_log(sprintf(
            'WVC Sync: Request sent to %s with body: %s',
            $endpoint_url,
            wp_json_encode($request_body)
        ));

        // For non-blocking requests, we can't check the response immediately
        // But we can schedule a follow-up check if needed
        if (is_wp_error($response)) {
            error_log('WVC Sync: Request failed - ' . $response->get_error_message());

            return $response;
        }

        return array('success' => true, 'message' => 'Sync request sent');
    }

}