<?php

namespace WVC\Includes\Theme;

/**
 * WVC Debug Menu Class
 *
 * This file contains functionality for a debug menu that is only visible
 * when the wvc_debug=1 parameter is present in the URL.
 *
 * @package    WVC_Theme
 * @subpackage Includes\Theme
 * @author     10Web
 * @since      1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Debug Menu Class
 *
 * Provides debug menu functionality for admin users
 */
class Debug_Menu
{
    /**
     * Constructor
     */
    public function __construct()
    {
        // Check if debug mode is enabled via URL parameter
        add_action('init', array($this, 'check_debug_mode'));

        // Add admin menu if in debug mode
        add_action('admin_menu', array($this, 'add_debug_menu'));

        // Register settings
        add_action('admin_init', array($this, 'register_debug_settings'));
    }

    /**
     * Check if debug mode is enabled
     */
    public function check_debug_mode()
    {
        // Check if the wvc_debug parameter is set to 1
        if (isset($_GET['wvc_debug'])) {
            if ($_GET['wvc_debug'] === '1') {
                // Only set debug mode for admin users
                if (current_user_can('manage_options')) {
                    // Set a transient to enable debug mode for this session
                    set_transient('wvc_debug_mode_enabled', true);
                }
            } else {
                delete_transient('wvc_debug_mode_enabled');
            }
        }
    }

    /**
     * Check if currently in debug mode
     *
     * @return bool True if debug mode is enabled
     */
    public function is_debug_mode()
    {
        // Check if the transient is set
        return (bool)get_transient('wvc_debug_mode_enabled');
    }

    /**
     * Add debug menu to admin menu
     */
    public function add_debug_menu()
    {
        // Only show the menu if in debug mode
        if ( ! $this->is_debug_mode()) {
            return;
        }

        // Add the main menu
        add_menu_page(
            'WVC Debug',           // Page title
            'WVC Debug',           // Menu title
            'manage_options',      // Capability
            'wvc-debug',           // Menu slug
            array($this, 'debug_menu_page'), // Callback
            'dashicons-warning',   // Icon
            99                     // Position
        );

        // Add submenu for configuration
        add_submenu_page(
            'wvc-debug',           // Parent slug
            'WVC Debug Config',    // Page title
            'Config',              // Menu title
            'manage_options',      // Capability
            'wvc-debug-config',    // Menu slug
            array($this, 'debug_config_page') // Callback
        );
    }

    /**
     * Register settings for the debug menu
     */
    public function register_debug_settings()
    {
        // Only register settings if in debug mode
        if ( ! $this->is_debug_mode()) {
            return;
        }

        register_setting('wvc_debug_config', 'wvc_debug_configs');
    }

    /**
     * Debug menu main page
     */
    public function debug_menu_page()
    {
        ?>
        <div class="wrap">
            <h1>WVC Debug</h1>
            <p>Welcome to the WVC Debug menu. This menu is only visible when the <code>wvc_debug=1</code> parameter is
                in the URL.</p>

            <h2>Debug Information</h2>
            <ul>
                <li>WVC Version: <?php echo WVC_VERSION; ?></li>
                <li>WordPress Version: <?php echo get_bloginfo('version'); ?></li>
                <li>PHP Version: <?php echo phpversion(); ?></li>
            </ul>

            <h2>Debug Options</h2>
            <ul>
                <li><a href="<?php echo admin_url('admin.php?page=wvc-debug-config'); ?>">Config Settings</a></li>
            </ul>
        </div>
        <?php
    }

    /**
     * Debug config page
     */
    public function debug_config_page()
    {
        // Get saved options
        $options = get_option('wvc_debug_configs', array(
            'api_url' => 'https://testwvc.10web.io'
        ));

        // Save settings if form was submitted
        if (isset($_POST['wvc_debug_config_submit'])) {
            check_admin_referer('wvc_debug_config_action', 'wvc_debug_config_nonce');

            $options['api_url'] = sanitize_text_field($_POST['wvc_api_url']);

            update_option('wvc_debug_configs', $options);

            echo '<div class="notice notice-success is-dismissible"><p>Settings saved.</p></div>';
        }

        ?>
        <div class="wrap">
            <h1>WVC Debug Configuration</h1>

            <form method="post" action="">
                <?php wp_nonce_field('wvc_debug_config_action', 'wvc_debug_config_nonce'); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="wvc_api_url">WVC API URL</label></th>
                        <td>
                            <select name="wvc_api_url" id="wvc_api_url">
                                <option value="https://testwvc.10web.io" <?php selected($options['api_url'],
                                    'https://testwvc.10web.io'); ?>>Test API
                                </option>
                                <option value="https://test1wvc.10web.io" <?php selected($options['api_url'],
                                    'https://test1wvc.10web.io'); ?>>Test1 API
                                </option>
                                <option value="https://devwvc.10web.io" <?php selected($options['api_url'],
                                    'https://devwvc.10web.io'); ?>>Dev API
                                </option>
                            </select>
                            <p class="description">Select which WVC API endpoint to use.</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="wvc_debug_config_submit" class="button-primary" value="Save Changes"/>
                </p>
            </form>
        </div>
        <?php
    }
}

// Initialize the debug menu
new Debug_Menu();