<?php

namespace WVC\Includes\Theme;

/**
 * WVC Theme Core Class
 *
 * This file contains the pure theme functionality including setup, assets,
 * widget registration, custom post types, and theme customizer features.
 *
 * @package    WVC_Theme
 * @subpackage Includes\Theme
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Theme Core Class
 *
 * Handles pure theme functionality including setup, assets, and features
 */
class Theme_Core
{

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Theme setup
        $this->theme_setup();

        // Enqueue scripts and styles
        $this->enqueue_assets();

        // Widget areas
        add_action("widgets_init", array($this, "widgets_init"));

        // Custom excerpt length
        add_filter("excerpt_length", array($this, "excerpt_length"));
        add_filter("excerpt_more", array($this, "excerpt_more"));

        // Custom fields
        add_action("add_meta_boxes", array($this, "add_custom_fields"));
        add_action("save_post", array($this, "save_custom_fields"));

        // Theme customizer
        add_action("customize_register", array($this, "customize_register"));

        // Admin bar functionality
        add_action('wp_before_admin_bar_render', array($this, 'add_wvc_edit_to_admin_bar'));
    }

    /**
     * Theme setup
     */
    public function theme_setup()
    {
        // Add theme support for various features

        // Enables SEO-friendly document titles managed by WordPress
        // This lets WordPress handle the <title> tag in the document head
        add_theme_support("title-tag");

        // Enables featured images (post thumbnails) for posts and pages
        // Required for setting featured images in posts/pages and using the_post_thumbnail() function
        add_theme_support("post-thumbnails");

        // Enables custom logo functionality in the customizer
        // Allows site owners to upload and display their logo through Appearance > Customize
        add_theme_support("custom-logo");

        // Enables custom header image support in the customizer
        // Allows site owners to set a custom header image through Appearance > Customize
        add_theme_support("custom-header");

        // Enables custom background color and image in the customizer
        // Allows site owners to set custom background colors/images through Appearance > Customize
        add_theme_support("custom-background");

        // Enables automatic RSS feed links in the document head
        // Adds RSS feed links for posts and comments automatically in the <head> section
        add_theme_support("automatic-feed-links");

        // Enables HTML5 markup for WordPress core features
        // Replaces traditional markup with semantic HTML5 elements for better standards compliance
        add_theme_support(
            "html5",
            array(
                "search-form",  // Uses HTML5 markup for search forms
                "comment-form", // Uses HTML5 markup for comment forms
                "comment-list", // Uses HTML5 markup for comment lists
                "gallery",      // Uses HTML5 markup for image galleries
                "caption"       // Uses HTML5 markup for image captions
            )
        );

        add_theme_support( 'woocommerce' );
        remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10 );
        remove_action( 'woocommerce_after_main_content',  'woocommerce_output_content_wrapper_end', 10 );

        // Register navigation menus
        register_nav_menus(
            array(
                "primary" => __("Primary Menu", "wvc-theme"),
                "footer"  => __("Footer Menu", "wvc-theme")
            )
        );
    }


    /**
     * Enqueue scripts and styles
     */
    public function enqueue_assets()
    {

        add_action('wp_enqueue_scripts', array($this, 'add_tailwind_styles'), 1);

        add_action('wp_enqueue_scripts', array($this, 'enqueue_theme_stylesheet'), 10);

        // Enqueue dynamic CSS variables from Style Kits AFTER Tailwind script
        add_filter('script_loader_tag', array($this, 'enqueue_css_variables'), 10, 3);

        add_action('wp_enqueue_scripts', array($this, 'enqueue_wvc_assets'), 9);

        add_action('wp_enqueue_scripts', array($this, 'enqueue_page_assets'), 10);

        add_action('wp_enqueue_scripts', array($this, 'enqueue_single_template_assets'), 10);
        add_action('wp_enqueue_scripts', array($this, 'enqueue_archive_template_assets'), 10);

        add_action('wp_enqueue_scripts', array($this, 'enqueue_header_footer_assets'), 10);


    }

    public function enqueue_wvc_assets()
    {

        if (is_admin()) {
            return;
        }
        global $post;
        $post_id = $post ? $post->ID : null;
        $term_id = null;
        $taxonomy = null;

        // For taxonomy archives, get the post type from the queried object
        if (is_tax()) {
            $queried_object = get_queried_object();
            if ($queried_object && isset($queried_object->taxonomy)) {
                // Get term_id from the queried object
                if (isset($queried_object->term_id)) {
                    $term_id = $queried_object->term_id;
                }
                $taxonomy = $queried_object->taxonomy;
            }
        } 
        wp_enqueue_script('wvc-js-utils', get_template_directory_uri() . '/assets/js/index.umd.js', array(), null,
            false);

        wp_localize_script('wvc-js-utils', 'WVC', \WVC\Core\Core::get_wvc_params_to_localize($post_id, $term_id, $taxonomy, false));

        wp_enqueue_script('wvc', get_template_directory_uri() . '/assets/js/wvc.js', array('wvc-js-utils'), null,
            false);

        wp_localize_script('wvc', 'WVCMenus', \WVC\Core\Core::get_menus_to_localize(null, false));

    }

    public function enqueue_theme_stylesheet()
    {
        // Enqueue theme stylesheet (style.css)
        wp_enqueue_style(
            "wvc-theme-style",
            get_stylesheet_uri(), // Path to style.css
            array(), // Dependencies
            wp_get_theme()->get('Version'), // Version from style.css header
            'all' // Media type
        );
    }

    public function add_tailwind_styles()
    {

        # we enqueue tailwindcss only if this is not a vibecode page, vibecode pages have their own tailwindcss
        # also vibecode templates (single and archive) have their own tailwindcss
        # however we enqueue it for regular pages even if they have vibecode header footer

        if (self::is_vibecode_singular() || self::is_vibecode_template()) {
            return;
        }

        wp_enqueue_script(
            "tailwindcss-browser",
            "https://cdn.jsdelivr.net/npm/@tailwindcss/browser@4",
            array(),
            null,
            false,
        );
    }

    /**
     * Enqueue page-specific assets
     */
    public function enqueue_page_assets()
    {
        if (is_admin()) {
            return;
        }

        if ( ! self::is_vibecode_singular()) {
            return;
        }

        global $post;
        $post_id = $post->ID;

        $css = get_post_meta($post_id, "_wvc_page_css", true);
        $js  = get_post_meta($post_id, "_wvc_page_js", true);

        if ($css) {
            wp_register_style('wvc-page-style', false);
            wp_enqueue_style('wvc-page-style');

            wp_add_inline_style("wvc-page-style", $css);
        }
        if ($js) {

            wp_register_script('wvc-page-script', '', array(), null, false);
            wp_enqueue_script('wvc-page-script');
            wp_add_inline_script("wvc-page-script", $js);
        }
    }

    /**
     * Enqueue page-specific assets
     */
    public function enqueue_header_footer_assets()
    {
        if (is_admin()) {
            return;
        }
        if (self::is_vibecode_singular() || self::is_vibecode_template()) {
            # we enqueue header/footer assets only if this is not a vibecode page or template
            return;
        }

        $header_template = get_posts(array(
            'post_type'      => 'wvc_template',
            'post_status'    => 'publish',
            'meta_query'     => array(
                array(
                    'key'     => '_wvc_template_type',
                    'value'   => 'header',
                    'compare' => '='
                )
            ),
            'posts_per_page' => 1,
            'orderby'        => 'ID',
            'order'          => 'ASC'
        ));
        if ($header_template) {
            $header_template_id = $header_template[0]->ID;
        } else {
            $header_template_id = 0;
        }

        $footer_template = get_posts(array(
            'post_type'      => 'wvc_template',
            'post_status'    => 'publish',
            'meta_query'     => array(
                array(
                    'key'     => '_wvc_template_type',
                    'value'   => 'footer',
                    'compare' => '='
                )
            ),
            'posts_per_page' => 1,
            'orderby'        => 'ID',
            'order'          => 'ASC'
        ));
        if ($footer_template) {
            $footer_template_id = $footer_template[0]->ID;
        } else {
            $footer_template_id = 0;
        }

        $header_css = get_post_meta($header_template_id, "_wvc_page_css", true);
        $header_js  = get_post_meta($header_template_id, "_wvc_page_js", true);

        $footer_css = get_post_meta($footer_template_id, "_wvc_page_css", true);
        $footer_js  = get_post_meta($footer_template_id, "_wvc_page_js", true);

        if ($header_css) {
            wp_register_style('wvc-header-style', false);
            wp_enqueue_style('wvc-header-style');

            wp_add_inline_style("wvc-header-style", $header_css);
        }
        if ($header_js) {

            wp_register_script('wvc-header-script', '', array(), null, false);
            wp_enqueue_script('wvc-header-script');
            wp_add_inline_script("wvc-header-script", $header_js);
        }

        if ($footer_js) {
            wp_register_script('wvc-footer-script', '', array(), null, false);
            wp_enqueue_script('wvc-footer-script');
            wp_add_inline_script("wvc-footer-script", $footer_js);
        }

        if ($footer_css) {
            wp_register_style('wvc-footer-style', false);
            wp_enqueue_style('wvc-footer-style');

            wp_add_inline_style("wvc-footer-style", $footer_css);
        }
    }

    /**
     * Enqueue single-specific assets
     */
    public function enqueue_single_template_assets()
    {
        if (is_admin()) {
            return;
        }

        if ( ! is_singular()){
            return;
        }
        
        // Get current post type
        $current_post_type = get_post_type();

        if(! self::is_there_a_vibecode_template( "single", $current_post_type)) {
            return;
        }

       # no single page template, each page is individual vibecoded content
        if ($current_post_type === 'page') {
            return;
        }

        // Try to get CPT-specific template using the new function
        $single_template = self::is_there_a_vibecode_template("single", $current_post_type);

        // Fallback to generic 'single' template for 'post' type if CPT-specific not found
        if ( ! $single_template && $current_post_type !== 'post') {
            $single_template = self::is_there_a_vibecode_template("single", "post");
        }

        $single_template_id = $single_template ? $single_template->ID : 0;

        $single_css = get_post_meta($single_template_id, "_wvc_page_css", true);
        $single_js  = get_post_meta($single_template_id, "_wvc_page_js", true);

        if ($single_js) {
            wp_register_script('wvc-single-script', '', array(), null, false);
            wp_enqueue_script('wvc-single-script');
            wp_add_inline_script("wvc-single-script", $single_js);
        }

        if ($single_css) {
            wp_register_style('wvc-single-style', false);
            wp_enqueue_style('wvc-single-style');

            wp_add_inline_style("wvc-single-style", $single_css);
        }
    }


    public function enqueue_archive_template_assets()
    {
        if (is_admin()) {
            return;
        }

        if( ! is_archive()){
            return;
        }

        $current_post_type = get_post_type();
        if ( ! self::is_there_a_vibecode_template("archive", $current_post_type)) {
            return;
        }

        $archive_template = self::is_there_a_vibecode_template("archive", $current_post_type);
        $archive_template_id = $archive_template ? $archive_template->ID : 0;

        $archive_css = get_post_meta($archive_template_id, "_wvc_page_css", true);
        $archive_js  = get_post_meta($archive_template_id, "_wvc_page_js", true);
        
        if ($archive_js) {
            wp_register_script('wvc-archive-script', '', array(), null, false);
            wp_enqueue_script('wvc-archive-script');
            wp_add_inline_script("wvc-archive-script", $archive_js);
        }

        if ($archive_css) {
            wp_register_style('wvc-archive-style', false);
            wp_enqueue_style('wvc-archive-style');

            wp_add_inline_style("wvc-archive-style", $archive_css);
        }
    }

    /**
     * Enqueue CSS Variables from Style Kits
     *
     * This function retrieves all published style kits and generates
     * CSS custom properties (CSS variables) for the frontend
     */
    public function enqueue_css_variables($tag, $handle, $src)
    {
        if ($handle != 'tailwindcss-browser') {
            return $tag;
        }

        // Get only published style kits (should be only one)
        $style_kits = get_posts(array(
            "post_type"   => "style_kit",
            "post_status" => "publish",
            "numberposts" => 1,  // Only get the active published kit
            "orderby"     => "date",
            "order"       => "DESC",
            "meta_query"  => array(
                array(
                    "key"     => "_wvc_css_variables",
                    "compare" => "EXISTS"
                )
            )
        ));

        // Get the active style kit
        $active_kit = null;

        if ( ! empty($style_kits)) {
            $active_kit = $style_kits[0]; // Get the first (and only) published kit
        } else {
            // Fallback: get any style kit if no published ones exist
            $all_style_kits = get_posts(array(
                "post_type"   => "style_kit",
                "post_status" => "any",
                "numberposts" => 1,
                "orderby"     => "date",
                "order"       => "DESC"
            ));

            if ( ! empty($all_style_kits)) {
                $active_kit = $all_style_kits[0];
            }
        }

        $tailwind_theme_stylesheet = "";

        if ( ! empty($active_kit)) {
            $tailwind_theme_stylesheet = get_post_meta($active_kit->ID, "_wvc_css_variables", true);
        }

        // Generate CSS variables with @theme structure
        $css_output = "";


        if ( ! empty($tailwind_theme_stylesheet)) {
            $css_output .= $tailwind_theme_stylesheet;
        } else {
            $css_output = $this->get_default_tailwind_theme_stylesheet();
        }

        $css_output .= "\n";

        // Escape the CSS output to prevent injection
        $css_output = wp_strip_all_tags($css_output);

        // Apply filter to allow customization of CSS output
        $css_output = apply_filters("wvc_css_variables_output", $css_output, array());

        # a trick - we attach inline style right after the script tag
        $tag .= "\n<style type='text/tailwindcss' id='wvc-tailwind-theme-stylesheet'>" . $css_output . "</style>\n";

        return $tag;
    }


    /**
     * Escape CSS values to prevent injection
     */
    private function escape_css_value($value)
    {
        // Remove any potential CSS injection attempts
        $value = str_replace(array(";", "}", "{", "/*", "*/"), "", $value);

        // Trim whitespace
        $value = trim($value);

        return $value;
    }

    /**
     * Get available CSS variables for frontend use
     *
     * @return array Array of CSS variables from all active style kits
     */
    public static function get_active_css_variables()
    {
        $style_kits = get_posts(array(
            "post_type"   => "style_kit",
            "post_status" => "publish",
            "numberposts" => 1,  // Only get the active published kit
            "orderby"     => "date",
            "order"       => "DESC",
            "meta_query"  => array(
                array(
                    "key"     => "_wvc_css_variables",
                    "compare" => "EXISTS"
                )
            )
        ));

        $all_variables       = array();
        $processed_variables = array();

        foreach ($style_kits as $kit) {
            $css_variables_json = get_post_meta($kit->ID, "_wvc_css_variables", true);
            $css_variables      = $css_variables_json ? json_decode($css_variables_json, true) : array();

            if ( ! empty($css_variables) && is_array($css_variables)) {
                foreach ($css_variables as $variable) {
                    if ( ! isset($variable["key"]) || ! isset($variable["value"])) {
                        continue;
                    }

                    $key = sanitize_text_field($variable["key"]);

                    // Skip if we already processed this variable (first kit wins)
                    if (isset($processed_variables[$key])) {
                        continue;
                    }

                    $all_variables[$key]       = $variable;
                    $processed_variables[$key] = true;
                }
            }
        }

        return $all_variables;
    }

    /**
     * Get the currently active (published) style kit
     *
     * @return WP_Post|null The active style kit post or null if none found
     */
    public static function get_active_style_kit()
    {
        $style_kits = get_posts(array(
            "post_type"   => "style_kit",
            "post_status" => "publish",
            "numberposts" => 1,
            "orderby"     => "date",
            "order"       => "DESC"
        ));

        return ! empty($style_kits) ? $style_kits[0] : null;
    }

    /**
     * Check if a style kit is currently active
     *
     * @param int $style_kit_id The style kit ID to check
     *
     * @return bool True if the style kit is active (published), false otherwise
     */
    public static function is_style_kit_active($style_kit_id)
    {
        $post = get_post($style_kit_id);

        return $post && $post->post_type === "style_kit" && $post->post_status === "publish";
    }

    /**
     * Activate a style kit (set to published, others to alternative)
     *
     * @param int $style_kit_id The style kit ID to activate
     *
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public static function activate_style_kit($style_kit_id)
    {
        $post = get_post($style_kit_id);

        if ( ! $post || $post->post_type !== "style_kit") {
            return new \WP_Error("invalid_style_kit", __("Invalid style kit ID.", "wvc-theme"));
        }

        // Update the post status to publish (this will trigger the transition hook)
        $result = wp_update_post(array(
            "ID"          => $style_kit_id,
            "post_status" => "publish"
        ));

        if (is_wp_error($result)) {
            return $result;
        }

        return true;
    }

    /**
     * Register widget areas
     */
    public function widgets_init()
    {
        register_sidebar(
            array(
                "name"          => __("Footer Widget Area", "wvc-theme"),
                "id"            => "footer-1",
                "description"   => __("Add widgets here for the footer.", "wvc-theme"),
                "before_widget" => "<div id=\"%1\$s\" class=\"footer-widget %2\$s\">",
                "after_widget"  => "</div>",
                "before_title"  => "<h3 class=\"footer-widget-title\">",
                "after_title"   => "</h3>"
            )
        );
    }

    /**
     * Custom post excerpt length
     */
    public function excerpt_length($length)
    {
        return 25;
    }

    /**
     * Custom excerpt more text
     */
    public function excerpt_more($more)
    {
        return "...";
    }

    /**
     * Add custom fields support
     */
    public function add_custom_fields()
    {
        add_meta_box(
            "wvc_custom_fields",
            "Custom Fields",
            array($this, "custom_fields_callback"),
            "post"
        );
    }

    /**
     * Custom fields callback
     */
    public function custom_fields_callback($post)
    {
        wp_nonce_field("wvc_save_custom_fields", "wvc_custom_fields_nonce");
        $featured_text = get_post_meta($post->ID, "_wvc_featured_text", true);
        echo "<p><label for=\"wvc_featured_text\">Featured Text:</label></p>";
        echo "<textarea id=\"wvc_featured_text\" name=\"wvc_featured_text\" rows=\"4\" cols=\"50\">" . esc_textarea($featured_text) . "</textarea>";
    }

    /**
     * Save custom fields
     */
    public function save_custom_fields($post_id)
    {
        if ( ! isset($_POST["wvc_custom_fields_nonce"]) || ! wp_verify_nonce(
                $_POST["wvc_custom_fields_nonce"],
                "wvc_save_custom_fields"
            )) {
            return;
        }

        if (defined("DOING_AUTOSAVE") && DOING_AUTOSAVE) {
            return;
        }

        if (isset($_POST["wvc_featured_text"])) {
            update_post_meta($post_id, "_wvc_featured_text", sanitize_textarea_field($_POST["wvc_featured_text"]));
        }
    }


    public static function is_vibecode_singular()
    {
        if (is_page() || is_front_page() || is_single()) {
            global $post;
            if ( ! $post) {
                return false;
            }
            $post_id = $post->ID;

            $wvc_editor_meta = \WVC\Core\Core::get_wvc_editor_meta($post_id);

            # if we have available vibecode template that can render the current post type, then we consider them as vibecode content
            if ($wvc_editor_meta) {
                return true;
            }
        }

        return false;
    }

    public static function is_vibecode_template(){
        if(is_singular()){
            // Get current post type
            $current_post_type = get_post_type();

            # pages are not vibecoded templates, but either vibecoded pages or regular pages
            if ($current_post_type == "page"){
                return false;
            }
            
            // Try to get CPT-specific template
            $single_template = self::is_there_a_vibecode_template("single", $current_post_type);
            
            // Fallback to generic "single" template for "post" type if CPT-specific not found
            if (!$single_template && $current_post_type !== "post") {
                $single_template = self::is_there_a_vibecode_template("single", "post");
            }
            
            return $single_template ? true : false;
        }
        elseif(is_archive()){
            // Get current post type for archive
            $current_post_type = get_post_type();
            
            // For taxonomy archives, get the post type from the queried object
            if (is_tax()) {
                $queried_object = get_queried_object();
                if ($queried_object && isset($queried_object->taxonomy)) {
                    $taxonomy = get_taxonomy($queried_object->taxonomy);
                    if ($taxonomy && !empty($taxonomy->object_type)) {
                        $current_post_type = $taxonomy->object_type[0];
                    }
                }
            }
            
            // For post type archives, get the post type from query var
            if (is_post_type_archive()) {
                $post_type_from_query = get_query_var("post_type");
                if (!empty($post_type_from_query)) {
                    $current_post_type = $post_type_from_query;
                }
            }
            
            // Try to get CPT-specific template
            $archive_template = self::is_there_a_vibecode_template("archive", $current_post_type);
            
            // Fallback to generic "archive" template for "post" type if CPT-specific not found
            if (!$archive_template && $current_post_type !== "post") {
                $archive_template = self::is_there_a_vibecode_template("archive", "post");
            }
            
            return $archive_template ? true : false;
        }
        else{
            return false;
        }
    }


    public static function is_there_a_vibecode_template($type = "single", $post_type = "post")
    {

        if($post_type === false){
            // special case when get_post_type() is not available ( taxonomy term contains no posts!)
            $queried_object = get_queried_object();
            if ($queried_object && isset($queried_object->taxonomy)) {
                $taxonomy = get_taxonomy($queried_object->taxonomy);
                if ($taxonomy && !empty($taxonomy->object_type)) {
                    $post_type = $taxonomy->object_type[0];
                }
            }
        }
        # there are no vibecode templates for pages, each page is individual vibecoded content
        if($post_type == "page"){
            return false;
        }
        // Validate that type is either "single" or "archive"
        if (!in_array($type, array("single", "archive"))) {
            return false;
        }
        
        // Validate that post_type exists and is registered
        if (!post_type_exists($post_type)) {
            return false;
        }

        if ($post_type != "post"){
            $template_type_value = $type.'_'.$post_type;
        }
        else{
            $template_type_value = $type;
        }
        

        $template = get_posts(array(
            'post_type'      => 'wvc_template',
            'post_status'    => 'publish',
            'meta_query'     => array(
                array(
                    'key'     => '_wvc_template_type',
                    'value'   => $template_type_value,
                    'compare' => '='
                )
            ),
            'posts_per_page' => 1,
            'orderby'        => 'ID',
            'order'          => 'ASC'
        ));


        return ($template) ? $template[0] : false;
    }


    /**
     * Theme customizer
     */
    public function customize_register($wp_customize)
    {
        // Add a section for theme options
        $wp_customize->add_section(
            "wvc_theme_options",
            array(
                "title"    => __("WVC Theme Options", "wvc-theme"),
                "priority" => 30
            )
        );

        // Add setting for header background color
        $wp_customize->add_setting(
            "wvc_header_bg_color",
            array(
                "default"           => "#ffffff",
                "sanitize_callback" => "sanitize_hex_color"
            )
        );

        $wp_customize->add_control(
            new \WP_Customize_Color_Control(
                $wp_customize,
                "wvc_header_bg_color",
                array(
                    "label"    => __("Header Background Color", "wvc-theme"),
                    "section"  => "wvc_theme_options",
                    "settings" => "wvc_header_bg_color"
                )
            )
        );
    }


    private function get_default_tailwind_theme_stylesheet()
    {
        return '

@theme {
  /* primary color: Chili Pepper (vibrant, warm, spicy)*/
  --color-primary-50: oklch(95% 0.04 50);   /* Very light, slightly warm */
  --color-primary-100: oklch(90% 0.05 48);  /* Lighter warm */
  --color-primary-200: oklch(82% 0.07 46);  /* Soft warm */
  --color-primary-300: oklch(74% 0.09 44);  /* Muted warm */
  --color-primary-400: oklch(65% 0.11 42);  /* Slightly desaturated warm */
  --color-primary-500: oklch(58% 0.14 40);  /* Base vibrant, warm red/orange */
  --color-primary-600: oklch(50% 0.16 38);  /* Deeper warm */
  --color-primary-700: oklch(42% 0.18 36);  /* Rich warm */
  --color-primary-800: oklch(34% 0.20 34);  /* Darker warm */
  --color-primary-900: oklch(26% 0.22 32);  /* Very dark warm */
  --color-primary-950: oklch(18% 0.23 30);  /* Deepest warm */

  /* secondary color: Agave (earthy, natural, grounding)*/
  --color-secondary-50: oklch(95% 0.03 140); /* Very light, slightly green */
  --color-secondary-100: oklch(90% 0.04 138);/* Lighter green */
  --color-secondary-200: oklch(82% 0.06 136);/* Soft green */
  --color-secondary-300: oklch(74% 0.08 134);/* Muted green */
  --color-secondary-400: oklch(65% 0.10 132);/* Slightly desaturated green */
  --color-secondary-500: oklch(55% 0.12 130);/* Base earthy green */
  --color-secondary-600: oklch(47% 0.14 128);/* Deeper green */
  --color-secondary-700: oklch(39% 0.16 126);/* Rich green */
  --color-secondary-800: oklch(31% 0.18 124);/* Darker green */
  --color-secondary-900: oklch(23% 0.20 122);/* Very dark green */
  --color-secondary-950: oklch(15% 0.21 120);/* Deepest green */
}';
    }

    /**
     * Add WVC Edit link to admin bar on front-end
     *
     * Adds a "WVC Edit" link to the WordPress admin bar when viewing
     * pages that are built with the WVC editor and removes the default "Edit Page" link
     */
    public function add_wvc_edit_to_admin_bar()
    {
        // Only show on front-end
        if (is_admin()) {
            return;
        }

        // Only show for logged-in users with edit permissions
        if ( ! current_user_can('edit_pages')) {
            return;
        }

        // Only show for logged-in users with edit permissions
        if ( ! current_user_can('edit_posts')) {
            return;
        }

        // Get current post ID
        $post_id = get_the_ID();
        if ( ! $post_id) {
            return;
        }

        // Check if current post is built with WVC editor
        $is_built_with_wvc = \WVC\Core\Core::is_built_with_wvc($post_id);
        $post_type = get_post($post_id)->post_type;
        
        if ( ! $is_built_with_wvc && !($post_type === 'post' or $post_type === 'product')) {
            return;
        }

        // Get the admin bar
        global $wp_admin_bar;
        if ( ! $wp_admin_bar) {
            return;
        }

        // Remove the default "Edit Page" link for WVC-built pages ( but not posts)
        if (!($post_type === 'post' or $post_type === 'product')) {
            $wp_admin_bar->remove_menu('edit');
        }

        // Create editor instance to get the URL
        $editor          = new \WVC\Editor\Editor(get_post($post_id));
        $custom_edit_url = $editor->get_edit_url();

        // Add the WVC Edit link to the admin bar
        $wp_admin_bar->add_menu(array(
            'id'    => 'wvc-edit',
            'title' => __('Edit with WVC Editor', 'wvc-theme'),
            'href'  => $custom_edit_url,
            'meta'  => array(
                'class' => 'wvc-edit-link',
                'title' => __('Edit with WVC Editor', 'wvc-theme')
            )
        ));
    }
}


// Global comment callback function
if ( ! function_exists("comment_callback")) {
    /**
     *
     * @param WP_Comment $comment The comment object
     * @param array $args The comment arguments
     * @param int $depth The depth of the comment
     */
    function comment_callback($comment, $args, $depth)
    {
        $tag             = ("div" === $args["style"]) ? "div" : "li";
        $comment_classes = empty($args["has_children"]) ? "mb-6" : "mb-6 parent";
        ?>
        <<?php echo $tag; ?> id="comment-<?php comment_ID(); ?>" <?php comment_class($comment_classes); ?>>
        <article id="div-comment-<?php comment_ID(); ?>"
                 class="comment-body bg-white rounded-lg border border-gray-500 overflow-hidden hover:border-gray-400 transition-colors duration-200">
            <footer class="comment-meta p-4 bg-primary-100 border-b border-primary-200">
                <div class="comment-author vcard flex items-start space-x-4">
                    <?php if ($args["avatar_size"] != 0) : ?>
                        <div class="comment-author-avatar flex-shrink-0">
                            <?php echo get_avatar($comment, $args["avatar_size"], "", "",
                                array("class" => "w-12 h-12 rounded-full border-2 border-primary-200")); ?>
                        </div>
                    <?php endif; ?>
                    <div class="comment-author-info flex-1 min-w-0">
                        <cite class="fn font-bold text-black text-lg block mb-1">
                            <?php echo get_comment_author_link(); ?>
                        </cite>
                        <div class="comment-meta-commentmetadata text-sm text-gray-600 flex items-center space-x-3">
                            <a href="<?php echo htmlspecialchars(get_comment_link($comment->comment_ID)); ?>"
                               class="hover:text-primary-600 transition-colors duration-200 flex items-center">
                                <svg class="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd"
                                          d="M6 2a1 1 0 00-1 1v1H4a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2h-1V3a1 1 0 10-2 0v1H7V3a1 1 0 00-1-1zm0 5a1 1 0 000 2h8a1 1 0 100-2H6z"
                                          clip-rule="evenodd"></path>
                                </svg>
                                <?php
                                printf(
                                    esc_html__("%1\$s at %2\$s", "wvc-theme"),
                                    get_comment_date(),
                                    get_comment_time()
                                );
                                ?>
                            </a>
                            <?php edit_comment_link(__("(Edit)", "wvc-theme"),
                                '<span class="text-gray-500 hover:text-primary-600 transition-colors duration-200">',
                                "</span>"); ?>
                        </div>
                    </div>
                </div>
            </footer>

            <div class="comment-content p-4 text-black leading-relaxed">
                <?php if ($comment->comment_approved == "0") : ?>
                    <div class="mb-4 p-3 bg-orange-100 border border-orange-200 rounded-lg">
                        <em class="comment-awaiting-moderation text-orange-800 text-sm font-medium flex items-center">
                            <svg class="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd"
                                      d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z"
                                      clip-rule="evenodd"></path>
                            </svg>
                            <?php esc_html_e("Your comment is awaiting moderation.", "wvc-theme"); ?>
                        </em>
                    </div>
                <?php endif; ?>
                <div class="prose prose-sm max-w-none text-gray-700">
                    <?php comment_text(); ?>
                </div>
            </div>

            <div class="reply p-4 bg-gray-50 border-t border-gray-200">
                <?php
                comment_reply_link(
                    array_merge(
                        $args,
                        array(
                            "add_below" => "div-comment",
                            "depth"     => $depth,
                            "max_depth" => $args["max_depth"],
                            "before"    => '<span class="reply-link inline-flex items-center px-4 py-2 bg-primary-500 text-white text-sm font-medium rounded-lg hover:bg-primary-600 transition-colors duration-200">',
                            "after"     => "</span>"
                        )
                    )
                );
                ?>
            </div>
        </article>
        <?php
    }
}
