<?php
/**
 * Custom Post Type Admin Columns and Filters
 *
 * Adds custom columns and filters to the WordPress admin list table for custom post types.
 * Automatically shows the most relevant fields based on field type and importance.
 *
 * @package    WVC_Theme
 * @subpackage Includes
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class WVC_CPT_Admin_Columns
 *
 * Handles admin list table customizations for custom post types
 */
class WVC_CPT_Admin_Columns
{
    /**
     * Instance of this class
     *
     * @var WVC_CPT_Admin_Columns
     */
    private static $instance = null;

    /**
     * Maximum number of custom columns to show
     *
     * @var int
     */
    const MAX_COLUMNS = 4;

    /**
     * Field types that are suitable for column display
     *
     * @var array
     */
    const COLUMN_FIELD_TYPES = array('text', 'email', 'date', 'number', 'select', 'checkbox');

    /**
     * Field types to exclude from columns (too long or complex)
     *
     * @var array
     */
    const EXCLUDE_FIELD_TYPES = array('textarea', 'url', 'image');

    /**
     * Field types that can be used as filters
     *
     * @var array
     */
    const FILTERABLE_TYPES = array('select', 'checkbox');

    /**
     * Field types that can be sortable
     *
     * @var array
     */
    const SORTABLE_TYPES = array('text', 'email', 'date', 'number');

    /**
     * Get singleton instance
     *
     * @return WVC_CPT_Admin_Columns
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        add_action('init', array($this, 'register_hooks'), 20);
    }

    /**
     * Register hooks for all custom post types
     */
    public function register_hooks()
    {
        $cpt_configs = get_option('wvc_custom_post_types', array());

        foreach ($cpt_configs as $post_type_key => $config) {
            // Only add columns/filters if CPT has custom fields
            $fields = get_option("wvc_cpt_fields_{$post_type_key}", array());

            if (empty($fields)) {
                continue;
            }

            // Register columns
            add_filter("manage_{$post_type_key}_posts_columns", array($this, 'add_custom_columns'), 10, 1);
            add_action("manage_{$post_type_key}_posts_custom_column", array($this, 'render_custom_column'), 10, 2);

            // Register sortable columns
            add_filter("manage_edit-{$post_type_key}_sortable_columns", array($this, 'add_sortable_columns'), 10, 1);

            // Register filters
            add_action('restrict_manage_posts', array($this, 'add_filters'), 10, 2);
        }

        // Apply custom sorting and filtering
        add_action('pre_get_posts', array($this, 'apply_custom_query'), 10, 1);
    }

    /**
     * Get relevant fields for column display
     *
     * @param string $post_type The post type key.
     *
     * @return array Array of field definitions suitable for columns
     */
    private function get_column_fields($post_type)
    {
        $fields = get_option("wvc_cpt_fields_{$post_type}", array());

        if (empty($fields)) {
            return array();
        }

        $column_fields = array();

        // First, prioritize required fields
        foreach ($fields as $field) {
            if (empty($field['required'])) {
                continue;
            }

            // Skip excluded types
            if (in_array($field['type'], self::EXCLUDE_FIELD_TYPES, true)) {
                continue;
            }

            // Only include suitable column types
            if (in_array($field['type'], self::COLUMN_FIELD_TYPES, true)) {
                $column_fields[] = $field;
            }

            // Stop if we reached max columns
            if (count($column_fields) >= self::MAX_COLUMNS) {
                return $column_fields;
            }
        }

        // Then add non-required fields if we have space
        foreach ($fields as $field) {
            if (!empty($field['required'])) {
                continue; // Already added above
            }

            // Skip excluded types
            if (in_array($field['type'], self::EXCLUDE_FIELD_TYPES, true)) {
                continue;
            }

            // Only include suitable column types
            if (in_array($field['type'], self::COLUMN_FIELD_TYPES, true)) {
                $column_fields[] = $field;
            }

            // Stop if we reached max columns
            if (count($column_fields) >= self::MAX_COLUMNS) {
                break;
            }
        }

        return $column_fields;
    }

    /**
     * Add custom columns to post list table
     *
     * @param array $columns Existing columns.
     *
     * @return array Modified columns
     */
    public function add_custom_columns($columns)
    {
        global $typenow;

        $column_fields = $this->get_column_fields($typenow);

        if (empty($column_fields)) {
            return $columns;
        }

        // Insert custom columns after title, before date
        $new_columns = array();
        foreach ($columns as $key => $value) {
            $new_columns[$key] = $value;

            // Insert after title
            if ($key === 'title') {
                foreach ($column_fields as $field) {
                    $column_key = "wvc_field_{$field['key']}";
                    $new_columns[$column_key] = $field['label'];
                }
            }
        }

        return $new_columns;
    }

    /**
     * Render custom column content
     *
     * @param string $column_name The column name.
     * @param int    $post_id     The post ID.
     */
    public function render_custom_column($column_name, $post_id)
    {
        // Check if this is our custom column
        if (strpos($column_name, 'wvc_field_') !== 0) {
            return;
        }

        // Extract field key from column name
        $field_key = str_replace('wvc_field_', '', $column_name);

        // Get post type and field schema
        $post_type = get_post_type($post_id);
        $fields = get_option("wvc_cpt_fields_{$post_type}", array());

        // Find field definition
        $field_def = null;
        foreach ($fields as $field) {
            if ($field['key'] === $field_key) {
                $field_def = $field;
                break;
            }
        }

        if (!$field_def) {
            echo '—';
            return;
        }

        // Get field value from individual meta key
        $meta_key = "wvc_{$post_type}_{$field_key}";
        $value = get_post_meta($post_id, $meta_key, true);

        // Format value based on field type
        echo $this->format_column_value($value, $field_def['type']);
    }

    /**
     * Format column value based on field type
     *
     * @param mixed  $value      The field value.
     * @param string $field_type The field type.
     *
     * @return string Formatted value for display
     */
    private function format_column_value($value, $field_type)
    {
        if (empty($value) && $value !== 0 && $value !== '0') {
            return '—';
        }

        switch ($field_type) {
            case 'checkbox':
                return !empty($value) ? __('Yes', 'wvc-theme') : __('No', 'wvc-theme');

            case 'date':
                if (empty($value)) {
                    return '—';
                }
                // Format date in WordPress date format
                $timestamp = strtotime($value);
                return $timestamp ? date_i18n(get_option('date_format'), $timestamp) : esc_html($value);

            case 'number':
                return number_format_i18n((float)$value);

            case 'email':
                return '<a href="mailto:' . esc_attr($value) . '">' . esc_html($value) . '</a>';

            case 'text':
            case 'select':
            default:
                // Truncate long text
                $max_length = 50;
                if (strlen($value) > $max_length) {
                    return esc_html(substr($value, 0, $max_length)) . '...';
                }
                return esc_html($value);
        }
    }

    /**
     * Add sortable columns
     *
     * @param array $columns Existing sortable columns.
     *
     * @return array Modified sortable columns
     */
    public function add_sortable_columns($columns)
    {
        global $typenow;

        $column_fields = $this->get_column_fields($typenow);

        if (empty($column_fields)) {
            return $columns;
        }

        foreach ($column_fields as $field) {
            // Only make certain field types sortable
            if (in_array($field['type'], self::SORTABLE_TYPES, true)) {
                $column_key = "wvc_field_{$field['key']}";
                $columns[$column_key] = $column_key;
            }
        }

        return $columns;
    }

    /**
     * Add filter dropdowns to admin list table
     *
     * @param string $post_type The post type.
     * @param string $which     Top or bottom of the table.
     */
    public function add_filters($post_type, $which)
    {
        // Only show filters at the top
        if ($which !== 'top') {
            return;
        }

        $fields = get_option("wvc_cpt_fields_{$post_type}", array());

        if (empty($fields)) {
            return;
        }

        // Add filters for select and checkbox fields
        foreach ($fields as $field) {
            if (!in_array($field['type'], self::FILTERABLE_TYPES, true)) {
                continue;
            }

            $this->render_field_filter($field, $post_type);
        }
    }

    /**
     * Render filter dropdown for a field
     *
     * @param array  $field     The field definition.
     * @param string $post_type The post type.
     */
    private function render_field_filter($field, $post_type)
    {
        $filter_name = "wvc_filter_{$field['key']}";
        $current_value = isset($_GET[$filter_name]) ? sanitize_text_field($_GET[$filter_name]) : '';

        echo '<select name="' . esc_attr($filter_name) . '" id="' . esc_attr($filter_name) . '">';
        echo '<option value="">' . esc_html($field['label']) . ': ' . __('All', 'wvc-theme') . '</option>';

        if ($field['type'] === 'select' && !empty($field['options'])) {
            // Dropdown for select fields with predefined options
            foreach ($field['options'] as $option) {
                $selected = ($current_value === $option) ? ' selected' : '';
                echo '<option value="' . esc_attr($option) . '"' . $selected . '>' . esc_html($option) . '</option>';
            }
        } elseif ($field['type'] === 'checkbox') {
            // Yes/No filter for checkbox fields
            echo '<option value="1"' . selected($current_value, '1', false) . '>' . __('Yes', 'wvc-theme') . '</option>';
            echo '<option value="0"' . selected($current_value, '0', false) . '>' . __('No', 'wvc-theme') . '</option>';
        }

        echo '</select>';
    }

    /**
     * Apply custom sorting and filtering to post query
     *
     * @param WP_Query $query The WordPress query object.
     */
    public function apply_custom_query($query)
    {
        // Only modify admin queries for our custom post types
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }

        $post_type = $query->get('post_type');
        if (empty($post_type)) {
            global $typenow;
            $post_type = $typenow;
        }

        // Check if this is a custom post type with fields
        $fields = get_option("wvc_cpt_fields_{$post_type}", array());
        if (empty($fields)) {
            return;
        }

        // Apply custom sorting
        $this->apply_custom_sorting($query, $post_type, $fields);

        // Apply custom filtering
        $this->apply_custom_filtering($query, $post_type, $fields);
    }

    /**
     * Apply custom sorting to query
     *
     * @param WP_Query $query      The query object.
     * @param string   $post_type  The post type.
     * @param array    $fields     The field definitions.
     */
    private function apply_custom_sorting($query, $post_type, $fields)
    {
        $orderby = $query->get('orderby');

        // Check if sorting by our custom column
        if (empty($orderby) || strpos($orderby, 'wvc_field_') !== 0) {
            return;
        }

        // Extract field key
        $field_key = str_replace('wvc_field_', '', $orderby);

        // Find field definition
        $field_def = null;
        foreach ($fields as $field) {
            if ($field['key'] === $field_key) {
                $field_def = $field;
                break;
            }
        }

        if (!$field_def || !in_array($field_def['type'], self::SORTABLE_TYPES, true)) {
            return;
        }

        // Sort by individual meta key
        $meta_key = "wvc_{$post_type}_{$field_key}";
        $query->set('meta_key', $meta_key);

        // For numeric fields, use numeric comparison
        if ($field_def['type'] === 'number') {
            $query->set('orderby', 'meta_value_num');
        } else {
            $query->set('orderby', 'meta_value');
        }
    }

    /**
     * Apply custom filtering to query
     *
     * @param WP_Query $query      The query object.
     * @param string   $post_type  The post type.
     * @param array    $fields     The field definitions.
     */
    private function apply_custom_filtering($query, $post_type, $fields)
    {
        $meta_query = array('relation' => 'AND');

        foreach ($fields as $field) {
            if (!in_array($field['type'], self::FILTERABLE_TYPES, true)) {
                continue;
            }

            $filter_name = "wvc_filter_{$field['key']}";

            if (!isset($_GET[$filter_name]) || $_GET[$filter_name] === '') {
                continue;
            }

            $filter_value = sanitize_text_field($_GET[$filter_name]);

            // Add meta query condition for individual meta key
            $meta_key = "wvc_{$post_type}_{$field['key']}";
            $meta_query[] = array(
                'key'     => $meta_key,
                'value'   => $filter_value,
                'compare' => '=',
            );
        }

        if (count($meta_query) > 1) {
            $query->set('meta_query', $meta_query);
        }
    }
}

// Initialize the CPT Admin Columns Handler
WVC_CPT_Admin_Columns::get_instance();
